/**
 *
 * 仮想デバイス抽象化Classのパッケージ AbstractDeviceList: AbstractDevice{}
 * todo::検索して見つからないデバイスはAjaxで動的ロード＞生成して返す
 * todo::各モジュールで生成されている構造体・定数をグローバル定義の構造体(オブジェクト)にリファクタリングする
 * todo::ブラケット記法＞ドット記法に統一
 */

/************************************************************************************
 *
 * 定数定義
 *
 ************************************************************************************/
//PROPの値を設定画面のUIから編集する場合の各プロパティの型
PublishUIProp_TYPE={
    "INT":"INT","FLOAT":"FLOAT","BOOL":"BOOL","STRING":"STRING",
    "ENUM":"ENUM",//列挙体のプルダウン選択(第三引数 value.stepが列挙体になる)
    "SEND_NULL":"SEND_NULL",//メソッドの呼び出しのみの用途
    "SEND_VAL":"SEND_VAL",//数値＋送信ボタン　メソッドの呼び出しのみの用途
    "STRUCT_READ":"STRUCT_READ",//構造体の表示のみ (Keigan_Motor.GetRotationState用)
    "KM_POSARRAY":"KM_POSARRAY",//SetPosition用引数の配列 Array[position radians ,speed rad/sec, duration Xsec]
    "KM_POSARRAY_DEG":"KM_POSARRAY_DEG",//SetPosition_deg用引数の配列 Array[position deg ,speed rpm , duration Xsec]
    "KM_LEDARRAY":"KM_LEDARRAY",//LEDの点灯状態・モーターのLED色 Array[MOTOR_LED_STATE,R,G,B 0-255]
    "CONFIRM":"CONFIRM"//確認ダイアログボタン 
    
};
//プロパティのUIへの表示場所
PublishUIProp_UI_SHOW_TYPE={
    "Normal":0,//通常
    "Advance":1,//advanceモードのみ表示
    "MotorRegisterSetting":2//モーターレジスタセッティング用 (KiganMotor専用)
};
//デバイスタイプ
AbstractDeviceList_DeviceType={"BLE":"BLE","Module":"Module","Robot":"Robot","Input":"Input"};

/************************************************************************************
 *
 * 仮想デバイスClass定義
 *
 ************************************************************************************/

AbstractDeviceList=(function(){
    "use strict";

    /************************************************************************************
     * section::仮想デバイス（基底クラス）
     *
     * ●コンストラクタ _AbstractDevice_base(option)
     *  @option 初期化時に上書きするプロパティ値（データ復元時に使用）
     *  @class_option デバイスのクラス毎のオプション（初期化時に自動的に内部で設定する AbstractDeviceList.ClassOptionsの値）
     * ●プロパティ
     *  UID(R)
     *  device_name(R)
     *　device_type(R)
     *  is_private(R) 内部生成用（主にバインダーを非公開にする）フラグ
     *  EventHandlerType
     *  EventHandler
     *  BinderList [{Binder}]:バインド先のプロパティのリスト（これを他のモジュールとバインドする）
     * ●メソッド
     *  ○GetDeviceStateVals　保存データ取得メソッド　@returns {{UID: *, device_name: undefined, status: boolean}}
     *  ○ Destory デバイスの廃棄（削除処理で使用 イベントを発行し、他のデバイスから参照を解除してもらいGCさせる）
     *
     * ●イベント
     *  ○ OnDestory(this) 廃棄時イベント
     *  ○ OnStatusChange (this) PROP.statusの値変化時
     *
     * ●PROP{ デバイスの状態格納プロパティ（基底）
     *   @UID:生成されたユニーク値（オブジェクトの識別用 デバイス間のバインドで使用）
     *   @device_name:このデバイスの名前（保存・復帰時に使用　クラス生成時にCreateDeviceForSearchから自動で設定される）
     *   @device_type:このデバイスの種類（保存・復帰時に使用　クラス生成時にCreateDeviceForSearchから自動で設定される）
     *   @status: 制御可不可状態（setter 変更時にOnStatusChangeイベント通知）
     *   @device_sub_name:(RW)このデバイスのサブ名（ユーザーが他の同名デバイスと見分けがつくように、重複しない固有名を生成）
     *   @layout_x:(RW)編集画面時にレイアウトに配置するモジュール座標X
     *   @layout_y:(RW)編集画面時にレイアウトに配置するモジュール座標Y
     *   @UIinfo:(R)デバイスの表示用名称等の情報
     * ●CAN_EDIT_PROP[{Name:PROP名,Type:データタイプ(CAN_EDIT_PROP_TYPE)}]
     *  設定画面のUIから編集可能なPROPのリスト
     ************************************************************************************/
    function _AbstractDevice_base(option,class_option){
        "use strict";
        var self=this;
        /**
         * イベントタイプ定数
         */
        this.EventHandlerType={"OnDestory": "OnDestory","OnStatusChange":"OnStatusChange","OnValChange":"OnValChange","OnError":"OnError"};
        this.EventHandler =new EventTarget();
        this.BinderList={};//このデバイスに存在するバインダーのリスト（継承クラスで定義）
        this._status=(option.status===true);
        this._UIinfo=UTL_clone(class_option.UIinfo);
        /**
         * デバイスの状態格納オブジェ(PROP) 保存時にこの値の中が保存される。
         * @type {{UID: *, device_name: undefined, status, status}}
         */
        this.PROP={
            UID: (option.UID===undefined)?UTL_DeviceInner_uuid():option.UID//info::仮想デバイスUID生成基点
            ,device_name:class_option.device_name
            ,device_type:class_option.DeviceType
            ,device_sub_name:option.device_sub_name?option.device_sub_name:''
            ,is_private:option.is_private?true:false
            ,layout_x:option.layout_x?option.layout_x:0
            ,layout_y:option.layout_y?option.layout_y:0
            ,device_ui_order:option.device_ui_order?option.device_ui_order:undefined//UIの並び順
            ,device_ui_size_px:option.device_ui_size_px?option.device_ui_size_px:{width:null,height:null}//UIのサイズ
        };
        this.PublishUIPropList=[];
        //--------------------//
        //  バインダーの生成
        //--------------------//
        this.BinderList={};

    }
    // -------------------//
    //  パブリックプロパティ
    //--------------------//
    Object.defineProperty(_AbstractDevice_base.prototype,'UID',{get:function() {return this.PROP.UID;}});
    Object.defineProperty(_AbstractDevice_base.prototype,'device_name',{get:function() {return this.PROP.device_name;}});
    Object.defineProperty(_AbstractDevice_base.prototype,'device_type',{get:function() {return this.PROP.device_type;}});
    Object.defineProperty(_AbstractDevice_base.prototype,'device_sub_name',{get:function() {return this.PROP.device_sub_name;},set:function(val){this.PROP.device_sub_name=val}});
    Object.defineProperty(_AbstractDevice_base.prototype,'is_private',{get:function() {return this.PROP.is_private;}});
    Object.defineProperty(_AbstractDevice_base.prototype,'layout_x',{get:function() {return this.PROP.layout_x;},set:function(val){this.PROP.layout_x=val}});
    Object.defineProperty(_AbstractDevice_base.prototype,'layout_y',{get:function() {return this.PROP.layout_y;},set:function(val){this.PROP.layout_y=val}});
    Object.defineProperty(_AbstractDevice_base.prototype,'device_ui_order',{get:function() {return this.PROP.device_ui_order;},set:function(val){this.PROP.device_ui_order=val}});
    Object.defineProperty(_AbstractDevice_base.prototype,'device_ui_size_px',{get:function() {return this.PROP.device_ui_size_px;},set:function(val){this.PROP.device_ui_size_px=val}});
    Object.defineProperty(_AbstractDevice_base.prototype,'UIinfo',{get:function() {return this._UIinfo;}});

    Object.defineProperty(_AbstractDevice_base.prototype, 'status', {
        get: function() {
            return this._status;
        },
        set:function(val){
            if(this._status !== val){
                this._status=val;
                this.EventHandler.fire({type: this.EventHandlerType.OnStatusChange, status:this._status,device:this});
            }
        }
    });

    /**
     * 保存データ取得メソッド
     * @returns {{UID: *, device_name: undefined, status, status}}
     * @constructor
     */
    _AbstractDevice_base.prototype.GetDeviceStateVals=function(){
        return this.PROP;
    };

    /**
     * エラー呼び出し
     * @param msg
     * @constructor
     */
    _AbstractDevice_base.prototype.CallError=function(msg){
        this.EventHandler.fire({type: this.EventHandlerType.OnError, msg:msg,device:this});
        UTL_LOG("ERR:"+msg+":"+this.device_name);
    };
    /**
     * デバイスの廃棄処理
     * @param device
     * @constructor
     */
    _AbstractDevice_base.prototype.Destory=function(){
        this.RemoveAllBinder();//デバイス内で生成した全バインダーの一括廃棄
        this.EventHandler.fire({type: this.EventHandlerType.OnDestory,UID:this.UID,device:this});
        AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnIconViewRemoveDrowCall,device:this});//UI再描画用
    };
    /**
     * 設定画面のUIから編集可能なプロパティを公開する
     * @param PropertyName
     * @param PropertyType PublishUIProp_TYPE
     * @param LowLimit
     * @param MaxLimit
     * @param Step 増減するステップ数（入力のUIのステッパーで使用）
     * @constructor
     */
    _AbstractDevice_base.prototype.PublishUIProp=function(args){
        var name=args.prop_name;
        var type=args.prop_type;
        var show_name=(args.prop_show_name!=undefined?args.prop_show_name:name);
        var lowlim=args.lowlim;
        var maxlim=args.maxlim;
        var confirm_msg=args.confirm_msg;
        var step=args.step;
        var ui_show_type=(args.ui_show_type!=undefined?args.ui_show_type:PublishUIProp_UI_SHOW_TYPE.Normal);
        var publishUIPropDelegateMethods={refresh:function(){}};//Deviceモジュール側からプロパティUIを操作する場合に提供されるUI側のメソッド。bindingHandler内で定義される

        // info::プロパティ説明の生成。各言語はlanguages_add_device.js内
        // 識別子は「 [デバイス名(device_name)]_[プロパティ名]_prop_info_txt」
	    // ex) WORD('ble_KeiganMotor_MeasurementNotify_prop_info_txt')
        var prop_info_txt=WORD(this.device_name+"_"+name+"_prop_info_txt");
        //info::UIPropListをグローバル定義の構造体として独立させる
        this.PublishUIPropList.push({name:name,type:type,show_name:show_name,lowlim:lowlim,maxlim:maxlim,step:step,ui_show_type:ui_show_type,prop_info_txt:prop_info_txt,publishUIPropDelegateMethods:publishUIPropDelegateMethods,confirm_msg:confirm_msg});
        return publishUIPropDelegateMethods;
    };
    /**
     * UIから編集可能なプロパティのリスト
     * @returns [{prop:プロパティ参照,name:PropertyName,type:PropertyType,lowlim:LowLimit,maxlim:MaxLimit},・]
     * @constructor
     */
    _AbstractDevice_base.prototype.GetPublishUIPropList=function(){
        return this.PublishUIPropList;
    };

    /**
     *section::バインダー生成メソッド
     * @param args:{binder_name,type_list,allow_list,dataflowcb_in,dataflowcb_get,is_dataflow_out,is_dataflow_pull,option}
     * args:{
     *      binder_name:String //参照バインダー名 (Binder_Root固定)
     *      type_list:DeviceValBinder_TYPE_LIST //自身の接続タイプのビットフラグ　データ種別(デバイスタイプDEVICE_XXX 変数型VAL_XXX データ方向DATA_XXX) は必ず指定(指定が無いとその種別は全て許可になる)
     *      allow_list:DeviceValBinder_TYPE_LIST //許可する接続タイプのビットフラグ 複数指定はORになる　　データ種別(デバイスタイプDEVICE_XXX 変数型VAL_XXX データ方向DATA_XXX) は必ず指定(指定が無いとその種別は全て許可になる)
     *      dataflowcb_in:function //変更時のCB function(val){}
     *      dataflowcb_get:function //取得時のCB function(){return val}
     *      is_dataflow_out:bool //このバインダーから接続先のバインダーへデータをPUSHするか　バインド時に属性として使用 実際のPUSHはDeviceValBinder.Par_Valを操作
     *      is_dataflow_pull:bool //このバインダーから接続先のバインダーのデータをGETするか　バインド時に属性として使用 実際の操作はDeviceValBinder.Par_Valを操作
     *      ui_show_type:DeviceValBinder_UI_SHOW_TYPE //UIに表示するバインダーのカテゴリ　通常とAdvanceタブ　DeviceValBinder_UI_SHOW_TYPE
     *      option:Object//バインドを復帰するオプション。クラス生成時のオプションを渡す。この中にoption["binder_"+binder_name]でbinding_binder_infoがある場合はそれを復帰
     *  }
     * @constructor
     */
    _AbstractDevice_base.prototype.CreateBinder=function(args){
        var the=this;
        var Binder=new DeviceValBinder({
            ValSettrCB:args.dataflowcb_in,
            ValGettrCB:args.dataflowcb_get,
            device_innerid:this.UID,//デバイスUID（オブジェクトの識別用 デバイス間のバインドで使用）
            binder_name:args.binder_name,
            binder_show_name:args.binder_show_name,
            // info::バインダー説明の生成。各言語はlanguages_add_device.js内 識別子は「 [デバイス名(device_name)]_[プロパティ名]_binder_info_txt」
            binder_info_txt:WORD(the.device_name+"_"+args.binder_name+"_binder_info_txt"),
            type_list:args.type_list,
            //データ接続方向タイプのbitをメソッドの定義・フラグの有無で自動生成
            dataflow_list:
                (args.dataflowcb_in?DeviceValBinder_DATA_FLOW_LIST.DATA_FLOW_IN:0)
                |(args.dataflowcb_get?DeviceValBinder_DATA_FLOW_LIST.DATA_FLOW_GET:0)
                |(args.is_dataflow_out?DeviceValBinder_DATA_FLOW_LIST.DATA_FLOW_OUT:0)
                |(args.is_dataflow_pull?DeviceValBinder_DATA_FLOW_LIST.DATA_FLOW_PULL:0),
            allow_Comparator:args.allow_list,
            binding_binder_info:args.option?args.option["binder_"+args.binder_name]:undefined,//バインド先情報
            is_private:args.option.is_private?true:false,
            ui_show_type:args.ui_show_type,
            owner:the
        });

        //バインド切り替え時のバインド先情報の保存
        Binder.EventHandler.addListener(Binder.EventHandlerType.OnBindChange,function(e){
            //UTL_LOG("OnBindChange:",e.binder.binder_name);
            the.PROP["binder_"+e.binder.binder_name]= e.binder.binding_binder_info;
        });
        the.BinderList[args.binder_name]=Binder;
        return Binder;
    };

    /**
     * デバイス内で生成した全バインダーの一括廃棄
     * @constructor
     */
    _AbstractDevice_base.prototype.RemoveAllBinder=function(){
        var self=this;
        Object.keys(this.BinderList).forEach(function (idx) {
            self.BinderList[idx].Destory();
        });
    };

    /**
     * バインダーのリスト（UIからのバインド用）
     * @returns [{name:バインダー名,binder:binder},・,・]
     * @constructor
     */
    _AbstractDevice_base.prototype.GetBinderList=function(){
        var self=this;
        var k= Object.keys(self.BinderList);
        var r=[];
        for(var i=0;i<k.length;i++){
            //r.push({name:k[i],binder:self.BinderList[k[i]]});
            r.push(self.BinderList[k[i]]);
        }
        return r;
    };



//----------------------------------------------------------------------------------------------------------------//
//  section::##### BLEデバイス抽象クラス
//----------------------------------------------------------------------------------------------------------------//

    /************************************************************************************
     * section::BLEデバイス（基底クラス）
     * ・保存を必要とするデバイスの状態はthis.PROPに格納する
     * ・イベントタイプ定数は基底クラスで定義済みなので右記の要領で追加定義する　ex)this.EventHandlerType["OnHoge"]="OnHoge"
     * ・PROP（保存するデバイスの状態格納プロパティ）は基底クラスで定義済み 追加定義 this.PROP.hoge="fuga"
     *  ●プロパティ
     *   device_uuid (R) BLEデバイスUUID
     *  ●イベント
     *  OnDestory　廃棄時（基底クラス）
     ************************************************************************************/
    function _AbstractDevice_BLE_base(option,class_option){
        "use strict";
        var self=this;
        _AbstractDevice_base.apply(this, arguments);

        //BLEデバイスUUID
        this.PROP.device_uuid=option.device_uuid?option.device_uuid:"";
        this.PROP.ble_device_name=option.ble_device_name?option.ble_device_name:"";
        //デバイス情報　ベンダー定義
        this.PROP.ble_manufacturer_name="";
        this.PROP.ble_hardware_revision="";
        this.PROP.ble_firmware_revision="";

        // -------------------//
        //section::イベントバインド
        //--------------------//
        //todo::デバイスの有効検知（デフォルトはfalse） this.status:bool
        ///破棄時のイベント（基底クラスから呼び出される）　
        this.EventHandler.addListener(this.EventHandlerType.OnDestory,function(e){

        });
        //デバイスのステータスが変わった場合
        this.EventHandler.addListener(this.EventHandlerType.OnStatusChange,function(e){
            if(e.status){
                //切断-->接続時
                self.bleGETDeviceInformation();
            }else{
                //接続-->切断時
            }
        });

    }
    //デバイス基底クラス継承
    _AbstractDevice_BLE_base.prototype = Object.create(_AbstractDevice_base.prototype,{constructor:{value:_AbstractDevice_BLE_base}});
    // -------------------//
    // section::定数定義
    //--------------------//
    //ベンダー定義デバイス情報　info::16bit表記でないと認識されない
    _AbstractDevice_BLE_base.prototype.GATTProfile_DeviceInformationService={
        "Service":"180a"
        ,"ManufacturerNameString":"2a29"
        ,"HardwareRevisionString":"2a27"
        ,"FirmwareRevisionString":"2a26"
    };
    // -------------------//
    //  パブリックプロパティ
    //--------------------//
    Object.defineProperty(_AbstractDevice_BLE_base.prototype,'device_uuid',{get:function() {return this.PROP.device_uuid;}});
    Object.defineProperty(_AbstractDevice_BLE_base.prototype,'ble_device_name',{get:function() {return this.PROP.ble_device_name;}});
    Object.defineProperty(_AbstractDevice_BLE_base.prototype,'ble_manufacturer_name',{get:function() {return this.PROP.ble_manufacturer_name;}});
    Object.defineProperty(_AbstractDevice_BLE_base.prototype,'ble_hardware_revision',{get:function() {return this.PROP.ble_hardware_revision;}});
    Object.defineProperty(_AbstractDevice_BLE_base.prototype,'ble_firmware_revision',{get:function() {return this.PROP.ble_firmware_revision;}});

    /**
     * ベンダー定義デバイス情報を纏めて取得
     */
    _AbstractDevice_BLE_base.prototype.bleGETDeviceInformation=function(){
        var self=this;
        BLECharacteristicController.Read(this.device_uuid, this.GATTProfile_DeviceInformationService.Service,this.GATTProfile_DeviceInformationService.ManufacturerNameString,
            function(data){
                self.PROP.ble_manufacturer_name=UTL_parseBLEStringPacket(data);
            },
            function(e){
            });
        BLECharacteristicController.Read(this.device_uuid, this.GATTProfile_DeviceInformationService.Service,this.GATTProfile_DeviceInformationService.HardwareRevisionString,
            function(data){
                self.PROP.ble_hardware_revision=UTL_parseBLEStringPacket(data);
            },
            function(e){
            });
        BLECharacteristicController.Read(this.device_uuid, this.GATTProfile_DeviceInformationService.Service,this.GATTProfile_DeviceInformationService.FirmwareRevisionString,
            function(data){
                self.PROP.ble_firmware_revision=UTL_parseBLEStringPacket(data);

            },
            function(e){
            });
    };



    /************************************************************************************
     * section::***** ble_KeiganMotor　Keigan Motor KM 1.00
         ■バインダー
             ◎Motor	(表示名:Motor)	//・モーターの制御ポート
             ◎SetSpeed 『Advance』	(表示名:SetSpeed rad)	//・速度を設定(rad / sec) 常用範囲 ±31.4
             ◎SetSpeed_rpm	(表示名:SetSpeed rpm)	//・速度を設定(rpm) 常用範囲 ±300
             ◎SetPosition 『Advance』	(表示名:SetPosition)	//・絶対の位置指定(rad)
             ◎SetStop	(表示名:SetStop)	//・モーターの停止 (int 0:停止 0以外:最大トルクでロック)
             ◎RotationStateNotify 『Advance』	(表示名:RotationState notify)
             ・モーターの位置・速度・トルクの定期通知
         ●プロパティ
             ◎ MeasurementNotify  『Advance』		//・モーターの回転情報を定期的に取得(ON/OFF) (RW: bool)
             ◎ GetRotationState		//・現在のモーター回転情報
             ◎ LedStateColor (RW:ST_MTLedState(MOTOR_LED_STATE,R,G,B 0-255) 『Advance』		//・ LEDの点灯状態・モーターのLED色
             ◎ SetSpeed_rpm	(W:「rpm 常用範囲 -300〜+300」)	//・速度を設定
             ◎ SetPosition  (W:[position,speed,duration])『Advance』		//・モーターの位置指定(rad,rad/sec)
             ◎ SetPosition_deg 『Advance』		//・モーターの位置指定(deg,rpm)
             ◎ SetRelPosition_deg  (W:[position,speed,duration])『Advance』		//・相対の位置指定(deg,rpm)
             ◎ PreSetPositon (W:radians)    『Advance』		//・現在の状態でモーターの絶対位置をセットする(rad)
             ◎ HardStop	 (W:bool)	                //・急停止-->ニュートラル
             ◎ HardBrake (W:bool)   『Advance』		//・急停止-->ロック
             ◎ IconLabel (RW: text) 『Advance』		//・編集のモーターに表示するアイコン
         ●イベント
             ◎ OnDestory(this) 廃棄時イベント (基底)
             ◎ OnLedStateColorChange(this) LED変更時
             ◎ OnMotorRotationStateChange //位置・速度・トルクのいずれかが変化した
             ◎ OnMeasurementNotifyChange //位置・速度・トルクの状態監視が変化した
     *
     ************************************************************************************/
    function _AbstractDevice_Keigan_Motor(option,class_option){
        "use strict";
        _AbstractDevice_BLE_base.apply(this,arguments);//_AbstractDevice_BLE_baseのコンストラクタ
        var self=this;

        //--------------------//
        //section::モーター用追加イベントハンドラ定義
        //--------------------//
        this.EventHandlerType["OnLedStateColorChange"]="OnLedStateColorChange";//LED変更時
        this.EventHandlerType["OnMeasurementNotifyChange"]="OnMeasurementNotifyChange";//位置・速度・トルクの状態監視が変化した
        this.EventHandlerType["OnMotorRotationStateChange"]="OnMotorRotationStateChange";//位置・速度・トルクのいずれかが変化した
        this.EventHandlerType["OnIconLabelChange"]="OnIconLabelChange";
        this.EventHandlerType["OnMotorRegisterValReceiveChange"]="OnMotorRegisterValReceiveChange";//モーターからいずれかのレジスター値を受信し内部プロパティに設定した
        this.EventHandlerType["OnMotorSaveAllRegistersComp"]="OnMotorSaveAllRegistersComp";//モーターのレジスタ値を保存が成功した
        //--------------------//
        //section::内部プロパティ
        //--------------------//
        this._LedStateColor=new ST_MTLedState(ST_MTLedState.MOTOR_LED_STATE.MOTOR_LED_STATE_ON_SOLID,0,0,0);
        this._Position=0;//絶対位置(radians)
        this._Speed=0;//速度(radians / second)
        this._Torque=0;//トルク(N・m)
        //モーターから受信したレジスタ値
        this._CurveType=this.MOTOR_CURVE_TYPE.CURVE_TYPE_NONE;
        this._Acc=0;
        this._Dec=0;
        this._SpeedP=0;
        this._SpeedI=0;
        this._PositionP=0;
        this._QcurrentP=0;
        this._QcurrentI=0;
        this._QcurrentD=0;
        //--------------------//
        //section::保存プロパティの復帰＆デフォルト値設定
        //--------------------//
       // this.PROP.MeasurementNotify=option.MeasurementNotify?true:false;//モーターの位置・速度・トルク等測定値の取得通知
        this.PROP.MeasurementNotify=true;//info::回転情報の監視Notifyは強制的にON
        this.PROP.SettingNotify=true;//info::レジスタ情報の監視Notifyは強制的にON
        this.PROP.IconLabel=option.IconLabel?option.IconLabel:'';
        //--------------------//
        //section::バインダーの生成
        //--------------------//
        //モーター制御プロトコルバインダー
        self.Binder_Motor=this.CreateBinder(
            {
                binder_name:"Motor"
                ,binder_show_name:'Motor'
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_MOTOR | DeviceValBinder_TYPE_LIST.VAL_CTL_MT //自身の接続タイプの配列  ,DeviceValBinder_TYPE_LIST.DEVICE_BLE,DeviceValBinder_TYPE_LIST.DEVICE_MOTOR
                ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_MOTOR^DeviceValBinder_TYPE_LIST.DEVICE_ALL)| DeviceValBinder_TYPE_LIST.VAL_CTL_MT
                ,dataflowcb_in:function(val){
                    //Commandに対応するプロパティを実行 (info:: UTL_IsSetterOnePerentProto：setterプロパティの動的指定時に、setterと間違えて関数や定数の参照を上書きするのを防止する為に使用)
                    if(val instanceof ST_CtlProtocol &&UTL_IsSetterOnePerentProto(self,val.Command)){
                        //command指定時にこの値があると、指定したIDのモーターのみ実行する
                        if(val.targetID&&val.targetID!==self.UID){
                            return;
                        }
                        self[val.Command]=val.Val;
                    }
                }
                ,dataflowcb_get:function(){
                    return new ST_NotifyProtocol("MTRotStat",new ST_MTRotState(self._Position,self._Speed,self._Torque),self.UID);
                }
                ,is_dataflow_out:true
                ,is_dataflow_pull:false
                ,ui_show_type:DeviceValBinder_UI_SHOW_TYPE.Normal
                ,option:option
            }
         );
        //バインド切り替え時
        self.Binder_Motor.EventHandler.addListener(self.Binder_Motor.EventHandlerType.OnBindChange,function(e){
            //LED情報の状態通知
            self.Binder_Motor.Par_Val=new ST_NotifyProtocol(self.EventHandlerType.OnLedStateColorChange,self.LedStateColor,self.UID);
        });

        ///速度制御  (W:float)(速度 単位：radians / second 常用範囲 -31.4〜+31.4 )
        self.Binder_SetSpeed=this.CreateBinder({
            binder_name:"SetSpeed"
            ,binder_show_name:'Run at Velocity'
            ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_MOTOR | DeviceValBinder_TYPE_LIST.VAL_FLOAT
            ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_MOTOR^DeviceValBinder_TYPE_LIST.DEVICE_ALL)|DeviceValBinder_TYPE_LIST.VAL_FLOAT | DeviceValBinder_TYPE_LIST.VAL_INT
            ,dataflowcb_in:function(val){ self.SetSpeed=val;}
            ,dataflowcb_get:null
            ,is_dataflow_out:false
            ,is_dataflow_pull:false
            ,ui_show_type:DeviceValBinder_UI_SHOW_TYPE.Advance
            ,option:option
        });
        ///速度制御 rpm換算 (W:float)(速度 単位：rpm 常用範囲 -300〜+300 )
        self.Binder_SetSpeed_rpm=this.CreateBinder({
            binder_name:"SetSpeed_rpm"
            ,binder_show_name:'Run at Velocity (rpm)'
            ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_MOTOR | DeviceValBinder_TYPE_LIST.VAL_FLOAT
            ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_MOTOR^DeviceValBinder_TYPE_LIST.DEVICE_ALL)|DeviceValBinder_TYPE_LIST.VAL_FLOAT | DeviceValBinder_TYPE_LIST.VAL_INT
            ,dataflowcb_in:function(val){self.SetSpeed_rpm=val;}
            ,dataflowcb_get:null
            ,is_dataflow_out:false
            ,is_dataflow_pull:false
            ,ui_show_type:DeviceValBinder_UI_SHOW_TYPE.Normal
            ,option:option
        });
        //位置制御
        self.Binder_SetPosition=this.CreateBinder({
            binder_name:"SetPosition"
            ,binder_show_name:'Move to Position'
            ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_MOTOR | DeviceValBinder_TYPE_LIST.VAL_VECTOR2 | DeviceValBinder_TYPE_LIST.VAL_VECTOR3 | DeviceValBinder_TYPE_LIST.VAL_MT_ROTSTATE
            // ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_MOTOR^DeviceValBinder_TYPE_LIST.DEVICE_ALL)|
            //             DeviceValBinder_TYPE_LIST.VAL_VECTOR2 |DeviceValBinder_TYPE_LIST.VAL_VECTOR3 |DeviceValBinder_TYPE_LIST.VAL_MT_ROTSTATE
            ,allow_list:DeviceValBinder_TYPE_LIST.VAL_VECTOR2 |DeviceValBinder_TYPE_LIST.VAL_VECTOR3 |DeviceValBinder_TYPE_LIST.VAL_MT_ROTSTATE
            ,dataflowcb_in:function(st_data){
                    self.SetPosition=st_data;
            }
            ,dataflowcb_get:null
            ,is_dataflow_out:false
            ,is_dataflow_pull:false
            ,ui_show_type:DeviceValBinder_UI_SHOW_TYPE.Advance
            ,option:option
        });

        // //todo::info::保留 (モーターのSetSoftStopの実装（級APIが未実装の為新APIで）)
        // //減速時間を指定して停止させる(rad^2 / s)
        // self.Binder_SetSoftStop=this.CreateBinder({
        //     binder_name:"SetSoftStop"
        //     ,binder_show_name:'SetSoftStop'
        //     ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_MOTOR | DeviceValBinder_TYPE_LIST.VAL_INT
        //     ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_MOTOR^DeviceValBinder_TYPE_LIST.DEVICE_ALL)|DeviceValBinder_TYPE_LIST.VAL_INT | DeviceValBinder_TYPE_LIST.VAL_FLOAT
        //     ,dataflowcb_in:function(val){
        //         this._SoftStop_val=val;
        //         this.bleMotorSoftStop(val);
        //     }
        //     ,dataflowcb_get:null
        //     ,is_dataflow_out:false
        //     ,is_dataflow_pull:false
        //     ,ui_show_type:DeviceValBinder_UI_SHOW_TYPE.Advance
        //     ,option:option
        // });

        //モーターの停止 (W:int 0:停止　0以外:最大トルクでロック)
        self.Binder_SetStop=this.CreateBinder({
            binder_name:"SetStop"
            ,binder_show_name:'SetStop'
            ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_MOTOR | DeviceValBinder_TYPE_LIST.VAL_BOOL
            ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_MOTOR^DeviceValBinder_TYPE_LIST.DEVICE_ALL)|DeviceValBinder_TYPE_LIST.VAL_NUMBER
            ,dataflowcb_in:function(val){
                if(val){
                    self.bleMotorHardBrake();
                }else{
                    self.bleMotorHardStop();
                }
            }
            ,dataflowcb_get:null
            ,is_dataflow_out:false
            ,is_dataflow_pull:false
            ,ui_show_type:DeviceValBinder_UI_SHOW_TYPE.Normal
            ,option:option
        });

        //モーターの位置・速度・トルクの定期通知(PUSHのみ)
        self.Binder_RotationStateNotify=this.CreateBinder({
            binder_name:"RotationStateNotify"
            ,binder_show_name:'Measurement Notification'
            ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_MOTOR | DeviceValBinder_TYPE_LIST.VAL_MT_ROTSTATE
            ,allow_list:DeviceValBinder_TYPE_LIST.DEVICE_ALL|DeviceValBinder_TYPE_LIST.VAL_MT_ROTSTATE
            ,dataflowcb_in:null
            ,dataflowcb_get:null
            ,is_dataflow_out:true
            ,is_dataflow_pull:false
            ,ui_show_type:DeviceValBinder_UI_SHOW_TYPE.Advance
            ,option:option
        });

        //モーターの位置定期通知(PUSHのみ)
        self.Binder_PositionNotify=this.CreateBinder({
            binder_name:"PositionNotify"
            ,binder_show_name:'Measurement Position Notification'
            ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_MOTOR | DeviceValBinder_TYPE_LIST.VAL_FLOAT
            ,allow_list:DeviceValBinder_TYPE_LIST.DEVICE_ALL|DeviceValBinder_TYPE_LIST.VAL_NUMBER
            ,dataflowcb_in:null
            ,dataflowcb_get:null
            ,is_dataflow_out:true
            ,is_dataflow_pull:false
            ,ui_show_type:DeviceValBinder_UI_SHOW_TYPE.Advance
            ,option:option
        });
        self.Binder_SpeedNotify=this.CreateBinder({
            binder_name:"SpeedNotify"
            ,binder_show_name:'Measurement Speed Notification'
            ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_MOTOR | DeviceValBinder_TYPE_LIST.VAL_FLOAT
            ,allow_list:DeviceValBinder_TYPE_LIST.DEVICE_ALL|DeviceValBinder_TYPE_LIST.VAL_NUMBER
            ,dataflowcb_in:null
            ,dataflowcb_get:null
            ,is_dataflow_out:true
            ,is_dataflow_pull:false
            ,ui_show_type:DeviceValBinder_UI_SHOW_TYPE.Advance
            ,option:option
        });
        self.Binder_TorqueNotify=this.CreateBinder({
            binder_name:"TorqueNotify"
            ,binder_show_name:'Measurement Torque Notification'
            ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_MOTOR | DeviceValBinder_TYPE_LIST.VAL_FLOAT
            ,allow_list:DeviceValBinder_TYPE_LIST.DEVICE_ALL|DeviceValBinder_TYPE_LIST.VAL_NUMBER
            ,dataflowcb_in:null
            ,dataflowcb_get:null
            ,is_dataflow_out:true
            ,is_dataflow_pull:false
            ,ui_show_type:DeviceValBinder_UI_SHOW_TYPE.Advance
            ,option:option
        });
        //--------------------//
        //section::初期化
        //--------------------//

        //ペアリングイベントは基底クラスで実装

        //--------------------//
        //section::編集用UIに公開するプロパティ
        //--------------------//
        this.PublishUIProp({prop_name:"GetRotationState",prop_show_name:"Measurement Values",prop_type:PublishUIProp_TYPE.STRUCT_READ,lowlim:null,maxlim:null,step:null,ui_show_type:PublishUIProp_UI_SHOW_TYPE.Normal});//
        this.PublishUIProp({prop_name:"LedStateColor",prop_show_name:"Led State and Color",prop_type:PublishUIProp_TYPE.KM_LEDARRAY,lowlim:null,maxlim:null,step:null,ui_show_type:PublishUIProp_UI_SHOW_TYPE.Advance});//

        this.PublishUIProp({prop_name:"SetSpeed_rpm",prop_show_name:"Run at Velocity (rpm)",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:-300,maxlim:300,step:0.1,ui_show_type:PublishUIProp_UI_SHOW_TYPE.Normal});
        this.PublishUIProp({prop_name:"SetPosition",prop_show_name:"Move to Position",prop_type:PublishUIProp_TYPE.KM_POSARRAY,lowlim:[-62.83185307179586,0,0],maxlim:[62.83185307179586,31.4,10],step:[0.017453292519943295,0.1,0.1],ui_show_type:PublishUIProp_UI_SHOW_TYPE.Advance});//

        this.PublishUIProp({prop_name:"SetPosition_deg",prop_show_name:"Move to Position (deg)",prop_type:PublishUIProp_TYPE.KM_POSARRAY_DEG,lowlim:[-3600,0,0],maxlim:[3600,300,10],step:[1,0.1,0.1],ui_show_type:PublishUIProp_UI_SHOW_TYPE.Advance});
        this.PublishUIProp({prop_name:"SetRelPosition_deg",prop_show_name:"Move by Distance (deg)",prop_type:PublishUIProp_TYPE.KM_POSARRAY_DEG,lowlim:[-3600,0,0],maxlim:[3600,300,10],step:[1,0.1,0.1],ui_show_type:PublishUIProp_UI_SHOW_TYPE.Advance});

        this.PublishUIProp({prop_name:"PreSetPositon",prop_show_name:"Preset Position",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:0,maxlim:6.283185307179586 ,step:0.017453292519943295,ui_show_type:PublishUIProp_UI_SHOW_TYPE.Advance});//radian角
       // this.PublishUIProp({prop_name:"SoftStop",prop_show_name:"SoftStop",prop_type:PublishUIProp_TYPE.SEND_VAL,lowlim:100,maxlim:5000,step:100,ui_show_type:PublishUIProp_UI_SHOW_TYPE.Advance});
        this.PublishUIProp({prop_name:"HardStop",prop_show_name:"Hard Stop",prop_type:PublishUIProp_TYPE.SEND_NULL,lowlim:null,maxlim:null,step:null,ui_show_type:PublishUIProp_UI_SHOW_TYPE.Normal});
        this.PublishUIProp({prop_name:"HardBrake",prop_show_name:"Hard Brake",prop_type:PublishUIProp_TYPE.SEND_NULL,lowlim:null,maxlim:null,step:null,ui_show_type:PublishUIProp_UI_SHOW_TYPE.Advance});
        this.PublishUIProp({prop_name:"IconLabel",prop_show_name:"Icon Label",prop_type:PublishUIProp_TYPE.STRING,lowlim:null,maxlim:null,step:null,ui_show_type:PublishUIProp_UI_SHOW_TYPE.Advance});

        //モーターレジスタ関係
        var CurveType_UIMethods= this.PublishUIProp({prop_name:"CurveType",prop_show_name:"Motor CurveType",prop_type:PublishUIProp_TYPE.ENUM,lowlim:null,maxlim:null,step:this.MOTOR_CURVE_TYPE,ui_show_type:PublishUIProp_UI_SHOW_TYPE.MotorRegisterSetting});

        var Acc_UIMethods= this.PublishUIProp({prop_name:"Acc",prop_show_name:"Motor Acc",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:0,maxlim:200,step:1,ui_show_type:PublishUIProp_UI_SHOW_TYPE.MotorRegisterSetting});
        var Dec_UIMethods= this.PublishUIProp({prop_name:"Dec",prop_show_name:"Motor Dec",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:0,maxlim:200,step:1,ui_show_type:PublishUIProp_UI_SHOW_TYPE.MotorRegisterSetting});


        var SpeedP_UIMethods= this.PublishUIProp({prop_name:"SpeedP",prop_show_name:"Motor SpeedPID P",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:5.000,maxlim:20,step:0.001,ui_show_type:PublishUIProp_UI_SHOW_TYPE.MotorRegisterSetting});
        var SpeedI_UIMethods= this.PublishUIProp({prop_name:"SpeedI",prop_show_name:"Motor SpeedPID I",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:0.0001,maxlim:0.01,step:0.0001,ui_show_type:PublishUIProp_UI_SHOW_TYPE.MotorRegisterSetting});
        var PositionP_UIMethods= this.PublishUIProp({prop_name:"PositionP",prop_show_name:"Motor PositionPID P",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:1.0,maxlim:20.0,step:0.0001,ui_show_type:PublishUIProp_UI_SHOW_TYPE.MotorRegisterSetting});

        var QcurrentP_UIMethods= this.PublishUIProp({prop_name:"QcurrentP",prop_show_name:"Motor qCurrentPID P",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:0.01,maxlim:3.0,step:0.01,ui_show_type:PublishUIProp_UI_SHOW_TYPE.MotorRegisterSetting});
        var QcurrentI_UIMethods= this.PublishUIProp({prop_name:"QcurrentI",prop_show_name:"Motor qCurrentPID I",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:0.001,maxlim:50.0,step:0.1,ui_show_type:PublishUIProp_UI_SHOW_TYPE.MotorRegisterSetting});
        var QcurrentD_UIMethods= this.PublishUIProp({prop_name:"QcurrentD",prop_show_name:"Motor qCurrentPID D",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:0,maxlim:0.01,step:0.0001,ui_show_type:PublishUIProp_UI_SHOW_TYPE.MotorRegisterSetting});

        this.PublishUIProp({prop_name:"ResetPID",prop_show_name:"Motor ResetPID",prop_type:PublishUIProp_TYPE.CONFIRM,lowlim:null,maxlim:null,step:null,confirm_msg:WORD('ble_KeiganMotor_ResetPID_confirm_msg'),ui_show_type:PublishUIProp_UI_SHOW_TYPE.MotorRegisterSetting});
        this.PublishUIProp({prop_name:"SaveAllRegisters",prop_show_name:"Motor SaveSettings",prop_type:PublishUIProp_TYPE.CONFIRM,lowlim:null,maxlim:null,step:null,confirm_msg:WORD('ble_KeiganMotor_SaveAllRegisters_confirm_msg'),ui_show_type:PublishUIProp_UI_SHOW_TYPE.MotorRegisterSetting});
        this.PublishUIProp({prop_name:"ResetAllRegisters",prop_show_name:"Motor ResetAllSettings",prop_type:PublishUIProp_TYPE.CONFIRM,lowlim:null,maxlim:null,step:null,confirm_msg:WORD('ble_KeiganMotor_ResetAllRegisters_confirm_msg'),ui_show_type:PublishUIProp_UI_SHOW_TYPE.MotorRegisterSetting});


        // -------------------//
        //section::イベントバインド
        //--------------------//
        //モーターからいずれかのレジスター値を受信し内部プロパティに設定した場合はUIをリフレッシュ
        this.EventHandler.addListener(this.EventHandlerType.OnMotorRegisterValReceiveChange,function(e){
            Acc_UIMethods.refresh();
            Dec_UIMethods.refresh();
            SpeedP_UIMethods.refresh();
            SpeedI_UIMethods.refresh();
            PositionP_UIMethods.refresh();
            CurveType_UIMethods.refresh();
            QcurrentP_UIMethods.refresh();
            QcurrentI_UIMethods.refresh();
            QcurrentD_UIMethods.refresh();
        });

        //モーターの接続（ステータス）が変わった場合
        this.EventHandler.addListener(this.EventHandlerType.OnStatusChange,function(e){
            console.log("モーターの接続（ステータス）が変わった:"+e.status);
            self.Binder_Motor.Par_Val=new ST_NotifyProtocol(self.EventHandlerType.OnStatusChange,e.status,self.UID);
            if(e.status){
                //切断-->接続時

                //info::重複しないLED色を自動生成して再設定（サブ名から判断して、モーター同士で重複しない色を検索）
                var sub_num=parseInt(self.device_sub_name,10);
                sub_num=isNaN(sub_num)|| self.MOTOR_LED_COLOR_TABLE.length <= sub_num?0:sub_num;

                self.LedStateColor= self.MOTOR_LED_COLOR_TABLE[sub_num].Clone();
                self.bleMotorSettingNotify(true);//info::モーターのレジスタ情報の取得通知の開始
                self.bleMotorMeasurementNotify(true);//info::bleのペアリング時に回転情報の監視のON
            }else{
                //接続-->切断時
                //info::接続時のモーター固有の色ble_device_name#以降に色を戻す
                if(String(self.ble_device_name).split('#').length===2){
                    var rgbst=self.ble_device_name.split('#')[1];
                    var rgb=URL_ConvartLedOptRGBCode(rgbst);
                    self.LedStateColor= new ST_MTLedState(ST_MTLedState.MOTOR_LED_STATE.MOTOR_LED_STATE_ON_SOLID,rgb[0],rgb[1],rgb[2]);
                }


               // self.LedStateColor= self.MOTOR_LED_COLOR_TABLE[sub_num].Clone();
            }

        });
        //回転情報の送信 dataflow_out
        this.EventHandler.addListener(this.EventHandlerType.OnMotorRotationStateChange,function(e){
            var rs=new ST_MTRotState(self._Position,self._Speed,self._Torque);
            self.Binder_Motor.Par_Val=new ST_NotifyProtocol(self.EventHandlerType.OnMotorRotationStateChange,rs,self.UID);
            self.Binder_RotationStateNotify.Par_Val=rs;

            self.Binder_PositionNotify.Par_Val=self._Position;
            self.Binder_SpeedNotify.Par_Val=self._Speed;
            self.Binder_TorqueNotify.Par_Val=self._Torque;
        });
        //LEDの変更時
        this.EventHandler.addListener(this.EventHandlerType.OnLedStateColorChange,function(e){

            self.Binder_Motor.Par_Val=new ST_NotifyProtocol(self.EventHandlerType.OnLedStateColorChange,self.LedStateColor,self.UID);
        });
        ///破棄時のイベント（基底クラスから呼び出される）　
        this.EventHandler.addListener(this.EventHandlerType.OnDestory,function(e){

        });
    ///
    }

    _AbstractDevice_Keigan_Motor.prototype = Object.create(_AbstractDevice_BLE_base.prototype,{constructor:{value:_AbstractDevice_Keigan_Motor}});

    // -------------------//
    // section::定数定義
    //--------------------//


    _AbstractDevice_Keigan_Motor.prototype.GATTProfile={
        "Service":"f140ea35-8936-4d35-a0ed-dfcd795baa8c"
        ,"KM_MOTOR_CONTROL":"f1400051-8936-4d35-a0ed-dfcd795baa8c"//モーターへの制御命令を取り扱う （旧）var
        ,"KM_MOTOR_CONTROL_MODE":"f1400052-8936-4d35-a0ed-dfcd795baa8c"//モーターの動作制御モードを取り扱う （旧）var
        ,"KM_MOTOR_MEASUREMENT":"f1400004-8936-4d35-a0ed-dfcd795baa8c"//モーターの位置・速度・トルク等測定値を取り扱う（旧新共通）var
        ,"KM_MOTOR_LED":"f1400003-8936-4d35-a0ed-dfcd795baa8c"//モーターのLEDを取り扱う（旧新共通）var
        ,"KM_MOTOR_SETTING":"f1400006-8936-4d35-a0ed-dfcd795baa8c"//（新）var モーターへの制御命令を取り扱う
    };
    //MOTOR_SETTINGのコマンド番号一覧
    _AbstractDevice_Keigan_Motor.prototype.KM_MOTOR_SETTING_COMMAND={
        "maxSpeed":0x02,
        "minSpeed":0x03,
        "curveType":0x05,
        "acc":0x07,
        "dec":0x08,
        "maxTorque":0x0E,
        "qCurrentP":0x18,
        "qCurrentI":0x19,
        "qCurrentD":0x1A,
        "speedP":0x1B,
        "speedI":0x1C,
        "positionP":0x1E,
        "resetPID":0x22,
        "limit":0x25,
        "pwmFrequency":0x26,
        "motorMeasurement":0x2C,
        "interface":0x2E,
        "logOutput":0x30,
        "ownColor":0x3A,
        "iMUMeasurement":0x3D,
        "read":0x40,//readRegister
        "saveAllRegisters":0x41,
        "reset":0x4E,
        "resetAllRegisters":0x4F
    };

    //モーター加減速曲線の種類を設定
    _AbstractDevice_Keigan_Motor.prototype.MOTOR_CURVE_TYPE={
        "CURVE_TYPE_NONE":0,//なし
        "CURVE_TYPE_TRAPEZOID":1// 線形加減速
       // "CURVE_TYPE_S_CURVE":2// S字加減速
    };
    //モーターのLEDカラーテーブル
    _AbstractDevice_Keigan_Motor.prototype.MOTOR_LED_COLOR_TABLE=[
        new ST_MTLedState(ST_MTLedState.MOTOR_LED_STATE.MOTOR_LED_STATE_ON_SOLID,255,0,0),//赤
        new ST_MTLedState(ST_MTLedState.MOTOR_LED_STATE.MOTOR_LED_STATE_ON_SOLID,0,255,0),//緑
        new ST_MTLedState(ST_MTLedState.MOTOR_LED_STATE.MOTOR_LED_STATE_ON_SOLID,0,0,255),//青
        new ST_MTLedState(ST_MTLedState.MOTOR_LED_STATE.MOTOR_LED_STATE_ON_SOLID,255,255,0),//黄色
        new ST_MTLedState(ST_MTLedState.MOTOR_LED_STATE.MOTOR_LED_STATE_ON_SOLID,255,0,255),//ピンク
        new ST_MTLedState(ST_MTLedState.MOTOR_LED_STATE.MOTOR_LED_STATE_ON_SOLID,0,255,255),//スカイブルー
        new ST_MTLedState(ST_MTLedState.MOTOR_LED_STATE.MOTOR_LED_STATE_ON_SOLID,84,255,0),//ライトグリーン
        new ST_MTLedState(ST_MTLedState.MOTOR_LED_STATE.MOTOR_LED_STATE_ON_SOLID,255,255,255)//白
    ];

    //-------------------//
    // section::パブリックプロパティ
    //--------------------//
    //LEDの点灯状態・モーターのLED色(RW:Array[MOTOR_LED_STATE,R,G,B 0-255])
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'LedStateColor',{
        get:function() {
            if(this._LedStateColor){return this._LedStateColor.Clone();}
        },
        set:function(_LedStateColor){
            this.bleMotorSetLedStateColor.apply(this,_LedStateColor.GetValArray());
        }
    });
    //LED情報をST_NotifyProtocolで1度送信する(編集画面でのLED表示更新用に、バインド先のモジュールからST_MTRotState経由で呼び出される)
    //info::呼び出しのみ
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'GetLedStateColorNotifyOne',{
        get:function() {
           return null;
        },
        set:function(val){
            //LED情報の状態通知
            this.Binder_Motor.Par_Val=new ST_NotifyProtocol(this.EventHandlerType.OnLedStateColorChange,this.LedStateColor,this.UID);
        }
    });


    //モーター加減速曲線の種類を設定(RW:Enum MOTOR_CURVE_TYPE)
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'CurveType',{
        get:function() {return this._CurveType;},
        set:function(val){
            this.bleMotorSetCurveType(val);
            this._CurveType=val;
        }
    });
    //モーターのレジスタ情報の監視ON/OFF(RW:bool)
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'SettingNotify',{
        get:function() {return this.PROP.SettingNotify;},
        set:function(flg){
            this.bleMotorSettingNotify(flg);
        }
    });
    //モーター状態の監視ON/OFF(RW:bool)
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'MeasurementNotify',{
        get:function() {return this.PROP.MeasurementNotify;},
        set:function(flg){
            this.bleMotorMeasurementNotify(flg);
        }
    });
    //位置・速度・トルクのまとめて取得プロパティ(R:object)
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'GetRotationState',{
        get:function() {
            return {position:this._Position,speed:this._Speed,torque:this._Torque};
        },
        set:function(){}
    });


    //速度制御送信(W:float)(速度 単位：radians / second 常用範囲 -31.4〜+31.4 )
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'SetSpeed',{
        get:function() {return 0;},
        set:function(radians_par_sec){
            this.bleMotorSetSpeed(radians_par_sec);
        }
    });
    //速度制御送信 rpm換算 (W:float)(速度 単位：rpm 常用範囲 -300〜+300 )
    //内部は(radians / second)なのでRPMから換算する
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'SetSpeed_rpm',{
        get:function() {return 0;},
        set:function(_rpm){
            var radians_par_sec=UTL_conv_rpm_to_radi_sec(_rpm);//(Math.PI*2/60)
            this.bleMotorSetSpeed(radians_par_sec);
        }
    });

    //絶対の位置指定
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'SetPosition',{
        get:function() {return null;},
        set:function(st_data){
            if(st_data==null){return;}
            var r_arg=[];
            switch (st_data.constructor){
                case ST_Vector2:
                    r_arg=[st_data.x,st_data.y,0];
                    break;
                case ST_Vector3:
                    r_arg=[st_data.x,st_data.y,st_data.z];
                    break;
                case ST_MTRotState:
                    r_arg=[st_data.Position,st_data.Speed,st_data.Torque];
                    break;
                case Array:
                    r_arg=st_data;
                    break;
            }
            this.bleMotorSetPosition.apply(this,r_arg);
        }
    });

    //絶対の位置指定 度・rpm 換算
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'SetPosition_deg',{
        get:function() {return null;},
        set:function(st_data){
            if(st_data==null){return;}
            var r_arg=[];
            switch (st_data.constructor){
                case ST_Vector2:
                    r_arg=[st_data.x,st_data.y,0];
                    break;
                case ST_Vector3:
                    r_arg=[st_data.x,st_data.y,st_data.z];
                    break;
                case ST_MTRotState:
                    r_arg=[st_data.Position,st_data.Speed,st_data.Torque];
                    break;
                case Array:
                    r_arg=st_data;
                    break;
            }
            //引数の 回転座標 度->radian(Math.PI*2/360)  速度 rpm ->radian/sec(Math.PI*2/60) に変換
            r_arg[0]=UTL_conv_deg_to_radi(r_arg[0]);
            r_arg[1]=UTL_conv_rpm_to_radi_sec(r_arg[1]);
            this.bleMotorSetPosition.apply(this,r_arg);


            // var na=[UTL_conv_deg_to_radi(arg_ar[0]),UTL_conv_rpm_to_radi_sec(arg_ar[1]),arg_ar[2]];
            // this.bleMotorSetPosition.apply(this,na);
        }
    });

    //相対の位置指定(W:[position radian,speed rad/sec,duration Xsec])
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'SetRelPosition',{
        get:function() {return null;},
        set:function(st_data){
            if(st_data==null){return;}
            var r_arg=[];
            switch (st_data.constructor){
                case ST_Vector2:
                    r_arg=[st_data.x,st_data.y,0];
                    break;
                case ST_Vector3:
                    r_arg=[st_data.x,st_data.y,st_data.z];
                    break;
                case ST_MTRotState:
                    r_arg=[st_data.Position,st_data.Speed,st_data.Torque];
                    break;
                case Array:
                    r_arg=st_data;
                    break;
            }
            this.bleMotorSetRelPosition.apply(this,r_arg);

            //this.bleMotorSetRelPosition.apply(this,arg_ar);
        }
    });

    //相対の位置指定 度 rpm 換算 (W:[position deg ,speed rpm , duration Xsec])
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'SetRelPosition_deg',{
        get:function() {return null;},
        set:function(st_data){
            if(st_data==null){return;}
            var r_arg=[];
            switch (st_data.constructor){
                case ST_Vector2:
                    r_arg=[st_data.x,st_data.y,0];
                    break;
                case ST_Vector3:
                    r_arg=[st_data.x,st_data.y,st_data.z];
                    break;
                case ST_MTRotState:
                    r_arg=[st_data.Position,st_data.Speed,st_data.Torque];
                    break;
                case Array:
                    r_arg=st_data;
                    break;
            }

            //引数の 回転座標 度->radian(Math.PI*2/360)  速度 rpm ->radian/sec(Math.PI*2/60) に変換
            r_arg[0]=UTL_conv_deg_to_radi(r_arg[0]);
            r_arg[1]=UTL_conv_rpm_to_radi_sec(r_arg[1]);
            this.bleMotorSetRelPosition.apply(this,r_arg);

            // //引数の 回転座標 度->radian(Math.PI*2/360)  速度 rpm ->radian/sec(Math.PI*2/60) に変換
            // var na=[UTL_conv_deg_to_radi(arg_ar[0]),UTL_conv_rpm_to_radi_sec(arg_ar[1]),arg_ar[2]];
            // // console.dir(arg_ar);
            // // console.dir(na);
            // this.bleMotorSetRelPosition.apply(this,na);
        }
    });

    //現在の状態でモーターの絶対位置をセットする(W:radians)
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'PreSetPositon',{
        get:function() {return null;},
        set:function(radians){
            //info::PreSetPositonは両方のモードで使用
            this.bleMotorPreSetPositon(radians);
        }
    });
    // //ソフトに減速し停止させる（減速時間を指定可能）(W:millisecond)
    // Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'SoftStop',{
    //     get:function() {
    //         return this._SoftStop_val;//_SoftStop_val 一時変数UIの入力値保持だけに使用
    //     },
    //     set:function(dec_duration_ms){
    //         this._SoftStop_val=dec_duration_ms;
    //        this.bleMotorSoftStop(dec_duration_ms);
    //     }
    // });
    //モーターをハードに減速し停止させる(W:bool)
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'HardStop',{
        get:function() {return false;},
        set:function(bool){//boolはダミー
            this.bleMotorHardStop();
        }
    });
    //モーターを最大トルクでハードに停止する (ロック)(W:bool)
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'HardBrake',{
        get:function() {return false;},
        set:function(bool){//boolはダミー
            this.bleMotorHardBrake();
        }
    });


    //絶対位置(R)(radians)
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'Position',{
        get:function() {return this._Position;},
        set:function (_radians) {}
    });
    //速度(R)(radians / second)
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'Speed',{
        get:function() {return this._Speed;},
        set:function (_radi_sec) {}
    });
    //トルク(R)(N・m)
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'Torque',{
        get:function() {return this._Torque;},
        set:function (_Nm) {}
    });
    //アイコンラベル
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'IconLabel', {
        get: function() {
            return this.PROP.IconLabel;
        },
        set:function(val){this.PROP.IconLabel=val;
           this.EventHandler.fire({type:this.EventHandlerType.OnIconLabelChange,IconLabel:val});//UI再描画用
            AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnIconViewReDrowCall,device:this});//アイコン再描画用
        }
    });


    //モーターの加速度
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'Acc', {
        get: function() {
            return this._Acc;
        },
        set:function(val){
            this.bleMotorSetPIDVal(this.KM_MOTOR_SETTING_COMMAND.acc,val);
            this._Acc=val;
        }
    });
    //モーターの原速度
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'Dec', {
        get: function() {
            return this._Dec;
        },
        set:function(val){
            this.bleMotorSetPIDVal(this.KM_MOTOR_SETTING_COMMAND.dec,val);
            this._Dec=val;
        }
    });
    //モーターの速度制御Pパラメタ
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'SpeedP', {
        get: function() {
            return this._SpeedP;
        },
        set:function(val){
            this.bleMotorSetPIDVal(this.KM_MOTOR_SETTING_COMMAND.speedP,val);
            this._SpeedP=val;
        }
    });
    //モーターの速度制御Iパラメタ
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'SpeedI', {
        get: function() {
            return this._SpeedI;
        },
        set:function(val){
            this.bleMotorSetPIDVal(this.KM_MOTOR_SETTING_COMMAND.speedI,val);
            this._SpeedI=val;
        }
    });
    //モーターの位置制御Pパラメタ
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'PositionP', {
        get: function() {
            return this._PositionP;
        },
        set:function(val){
            this.bleMotorSetPIDVal(this.KM_MOTOR_SETTING_COMMAND.positionP,val);
            this._PositionP=val;
        }
    });
    //モーターのq軸電流制御P
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'QcurrentP', {
        get: function() {
            return this._QcurrentP;
        },
        set:function(val){
            this.bleMotorSetPIDVal(this.KM_MOTOR_SETTING_COMMAND.qCurrentP,val);
            this._QcurrentP=val;
        }
    });
    //モーターのq軸電流制御I
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'QcurrentI', {
        get: function() {
            return this._QcurrentI;
        },
        set:function(val){
            this.bleMotorSetPIDVal(this.KM_MOTOR_SETTING_COMMAND.qCurrentI,val);
            this._QcurrentI=val;
        }
    });
    //モーターのq軸電流制御D
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'QcurrentD', {
        get: function() {
            return this._QcurrentD;
        },
        set:function(val){
            this.bleMotorSetPIDVal(this.KM_MOTOR_SETTING_COMMAND.qCurrentD,val);
            this._QcurrentD=val;
        }
    });
    //モーターのPIDパラメタを全てリセットする(W:bool)
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'ResetPID',{
        get:function() {return false;},
        set:function(bool){//boolはダミー
            this.bleMotorResetPID();
        }
    });
    //モーターのレジスタ値を保存する(W:bool)
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'SaveAllRegisters',{
        get:function() {return false;},
        set:function(bool){//boolはダミー
            this.bleMotorSaveAllRegisters();
        }
    });
    //全てのレジスタ値を工場出荷状態にする(W:bool)
    Object.defineProperty(_AbstractDevice_Keigan_Motor.prototype,'ResetAllRegisters',{
        get:function() {return false;},
        set:function(bool){//boolはダミー
            this.bleMotorResetAllRegisters();
        }
    });
    /**
     * ペアリング先のモーターを変更した場合に仮想デバイスの設定情報を再設定する
     */
    _AbstractDevice_Keigan_Motor.prototype.bleMotorRefresh=function(){
        this.LedStateColor=this.LedStateColor;
        this.MeasurementNotify=this.MeasurementNotify;
        this.SettingNotify=this.SettingNotify;
    };


    // -------------------//
    //  プライベート
    //--------------------//


    /**
     * モーターのLED状態・カラーをセットする
     * @param motorLightState  MOTOR_LED_STATE
     * @param colorR int 0-255
     * @param colorG int 0-255
     * @param colorB int 0-255
     */
    _AbstractDevice_Keigan_Motor.prototype.bleMotorSetLedStateColor=function(ledState,colorR,colorG,colorB){
        var self=this;
        var buffer = new ArrayBuffer(4);
        new DataView(buffer).setUint8(0,ledState);
        new DataView(buffer).setUint8(1,colorR);
        new DataView(buffer).setUint8(2,colorG);
        new DataView(buffer).setUint8(3,colorB);
        self._LedStateColor=new ST_MTLedState(ledState,colorR,colorG,colorB);

        //var debug_bin=new Uint8Array(buffer);//for debug
        BLECharacteristicController.Send(this.device_uuid, this.GATTProfile.Service, this.GATTProfile.KM_MOTOR_LED,buffer,
            function(e){
                //info::成功時でもCBには返り値が無い為、先のbuffer設定値を使用
                var rval = new Uint8Array(buffer);//[点灯状態,R,G,B]
                self._LedStateColor=new ST_MTLedState(rval[0],rval[1],rval[2],rval[3]);
                self.EventHandler.fire({type: self.EventHandlerType.OnLedStateColorChange, target:self});
                //AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnIconViewLedStateReDrowCall,device:self});//LEDアイコン再描画用
                AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnIconViewReDrowCall,device:self});//アイコン再描画用
            },
            function(e){
                self.CallError("bleMotorSetLedStateColor("+[ledState,colorR,colorG,colorB]+") Device_Keigan_Motor:"+self.device_uuid);
            }
        );
    };


    /**
     * モーターのLED状態・カラーを取得する
     *   info::コールバックなので、返り値が返せない CBのみ
     * 　info::呼び出すと必ずOnLedStateColorChangeが発火する
     * @param cb 取得完了時のCB function(LedStateColor){}
     */
    _AbstractDevice_Keigan_Motor.prototype.bleMotorGetLedStateColor=function(cb){
        var self=this;
        BLECharacteristicController.Read(this.device_uuid, this.GATTProfile.Service, this.GATTProfile.KM_MOTOR_LED,
            function(data){
                var rval = new Uint8Array(data);//[点灯状態,R,G,B]
                var nv=new ST_MTLedState(rval[0],rval[1],rval[2],rval[3]);
                var is_change= !self._LedStateColor.EQ(nv);
                self._LedStateColor=nv;
                if(typeof cb==="function"){
                    cb(self.LedStateColor.Clone());
                }
                if(is_change){
                    self.EventHandler.fire({type: self.EventHandlerType.OnLedStateColorChange, target:self});
                }
            },
            function(e){
                self.CallError("bleMotorGetLedStateColor(cb) Device_Keigan_Motor:"+self.device_uuid);
            });
        return ;
    };

    /**
     * モーターの速度をセットする 逆転正転
     * @param targetSpeed 速度 (radians/second)  常用範囲[-628.0〜628.0]
     */
    _AbstractDevice_Keigan_Motor.prototype.bleMotorSetSpeed=function(speed){
        var self=this;
        //BLEコマンドの連続送信防止
        if(self["isbleMotorSetSpeedExec"]){
            return;
        }else{
            self["isbleMotorSetSpeedExec"]=true;
            setTimeout(function(){self["isbleMotorSetSpeedExec"]=false;},50);
        }

        var buffer = new ArrayBuffer(5);
        new DataView(buffer).setUint8(0,0x10);//1byte目はコマンド
        new DataView(buffer).setFloat32(1, speed, false);//浮動小数を4byte変換 バイトオーダーはビッグエンディアン

        //var debug_bin=new Uint8Array(buffer);//for debug

        BLECharacteristicController.Send(this.device_uuid, this.GATTProfile.Service, this.GATTProfile.KM_MOTOR_CONTROL,buffer,
            function(e){
                //info::モーターの状態はMeasurementNotifyで取得
                // self._Speed=speed;
                // self.EventHandler.fire({type: self.EventHandlerType.OnSpeedChange, target:self});
            },
            function(e){
                self.CallError("bleMotorSetSpeed("+speed+") Device_Keigan_Motor"+self.device_uuid);
            });
    };


    /**
     * モーターの絶対位置をセットする（速度・加減速時間を指定）
     * @param targetPosition 角度(radians)　浮動小数4byte
     * @param speed 速度 (radians/second)  常用範囲[-628.0〜628.0]
     * @param duration 加減速時間(second)   加速・減速[+X 〜 -X] todo::ここの挙動確認（どう影響する？）
     */
    _AbstractDevice_Keigan_Motor.prototype.bleMotorSetPosition=function(targetPosition,speed,duration){
        var self=this;

        //BLEコマンドの連続送信防止
        if(self["isbleMotorSetPositionExec"]){
            return;
        }else{
            self["isbleMotorSetPositionExec"]=true;
            setTimeout(function(){self["isbleMotorSetPositionExec"]=false;},50);
        }

        //console.log("targetPosition:"+targetPosition,"speed:"+speed);
        var buffer = new ArrayBuffer(13);
        new DataView(buffer).setUint8(0,0x21);//コマンド
        new DataView(buffer).setFloat32(1, targetPosition, false);//ビッグエンディアン
        new DataView(buffer).setFloat32(5, Math.abs(speed), false);
        new DataView(buffer).setFloat32(9, duration, false);
        //var debug_bin=new Uint8Array(buffer);//for debug
        BLECharacteristicController.Send(this.device_uuid, this.GATTProfile.Service, this.GATTProfile.KM_MOTOR_CONTROL,buffer,
            function(e){
                //info::モーターの状態はMeasurementNotifyで取得
                //self._Speed=speed;
                // self._Position=targetPosition;
               // self.EventHandler.fire({type: self.EventHandlerType.OnPositionChange, target:self});
            },
            function(e){
                self.CallError("bleMotorSetPosition("+[targetPosition,speed,duration]+") Device_Keigan_Motor "+self.device_uuid);
            });
    };



    /**
     * モーターの相対位置をセットする（現在の位置からの差分±） （速度・加速時間指定可能)
     * @param relativePosition 角度(radians)　浮動小数4byte
     * @param speed 速度 (radians/second)  常用範囲[-628.0〜628.0]
     * @param duration 加減速時間(second)   加速・減速[+X 〜 -X] todo::ここの挙動確認（どう影響する？）
     */
    _AbstractDevice_Keigan_Motor.prototype.bleMotorSetRelPosition=function(relativePosition,speed,duration){
        var self=this;

        //BLEコマンドの連続送信防止
        if(self["isbleMotorSetRelPositionExec"]){
            return;
        }else{
            self["isbleMotorSetRelPositionExec"]=true;
            setTimeout(function(){self["isbleMotorSetRelPositionExec"]=false;},50);
        }

        var buffer = new ArrayBuffer(13);
        new DataView(buffer).setUint8(0,0x24);//コマンド
        new DataView(buffer).setFloat32(1, relativePosition, false);//ビッグエンディアン
        new DataView(buffer).setFloat32(5, Math.abs(speed), false);
        new DataView(buffer).setFloat32(9, duration, false);
       // console.log(relativePosition,Math.abs(speed),duration);
        //var debug_bin=new Uint8Array(buffer);//for debug
        BLECharacteristicController.Send(this.device_uuid, this.GATTProfile.Service, this.GATTProfile.KM_MOTOR_CONTROL,buffer,
            function(e){
                //info::Speed PositionはモーターからのNotifyによる値で反映
                //self._Speed=speed;
               // self._Position=relativePosition;
            },
            function(e){
                self.CallError("bleMotorSetRelPosition("+[relativePosition,speed,duration]+") Device_Keigan_Motor "+self.device_uuid);
            });
    };
    /**
     * 現在の状態でモーターの絶対位置をセットする
     * @param currentPosition 角度(radians)　浮動小数4byte
     */
    _AbstractDevice_Keigan_Motor.prototype.bleMotorPreSetPositon=function(currentPosition){
        var self=this;
        //BLEコマンドの連続送信防止
        if(self["isbleMotorPreSetPositonExec"]){
            return;
        }else{
            self["isbleMotorPreSetPositonExec"]=true;
            setTimeout(function(){self["isbleMotorPreSetPositonExec"]=false;},50);
        }

        var buffer = new ArrayBuffer(5);
        new DataView(buffer).setUint8(0,0x27);//コマンド
        new DataView(buffer).setFloat32(1, currentPosition, false);//ビッグエンディアン
        //var debug_bin=new Uint8Array(buffer);//for debug
        BLECharacteristicController.Send(this.device_uuid, this.GATTProfile.Service, this.GATTProfile.KM_MOTOR_CONTROL,buffer,
            function(e){
                //info::モーターの状態はMeasurementNotifyで取得

            },
            function(e){
                self.CallError("bleMotorSetRelPosition("+currentPosition+") Device_Keigan_Motor "+self.device_uuid);
            });
    };
    /**
     * 現在のトルクを設定する
     * @param targetTorque トルク(N・m)　浮動小数4byte
     */
    _AbstractDevice_Keigan_Motor.prototype.bleMotorSetTorque=function(targetTorque){
        var self=this;

        //BLEコマンドの連続送信防止
        if(self["isbleMotorSetTorqueExec"]){
            return;
        }else{
            self["isbleMotorSetTorqueExec"]=true;
            setTimeout(function(){self["isbleMotorSetTorqueExec"]=false;},50);
        }

        var buffer = new ArrayBuffer(5);
        new DataView(buffer).setUint8(0,0x70);//コマンド
        new DataView(buffer).setFloat32(1, currentPosition, false);//ビッグエンディアン
        //var debug_bin=new Uint8Array(buffer);//for debug
        BLECharacteristicController.Send(this.device_uuid, this.GATTProfile.Service, this.GATTProfile.KM_MOTOR_CONTROL,buffer,
            function(e){
                //info::モーターの状態はMeasurementNotifyで取得

            },
            function(e){
                self.CallError("bleMotorSetTorque("+targetTorque+") Device_Keigan_Motor "+self.device_uuid);
            });
    };

    /**
     * モーターをソフトに減速し停止させる（減速時間を指定可能）
     * @param dec_duration_ms 減速時間 単位：msec　浮動小数4byte
     */
    _AbstractDevice_Keigan_Motor.prototype.bleMotorSoftStop=function(dec_duration_ms){
        var self=this;
        var buffer = new ArrayBuffer(5);
        new DataView(buffer).setUint8(0,0x80);//コマンド
        new DataView(buffer).setFloat32(1, dec_duration_ms, false);//ビッグエンディアン
        //var debug_bin=new Uint8Array(buffer);//for debug
        BLECharacteristicController.Send(this.device_uuid, this.GATTProfile.Service, this.GATTProfile.KM_MOTOR_CONTROL,buffer,
            function(e){
                //info::モーターの状態はMeasurementNotifyで取得

            },
            function(e){
                self.CallError("bleMotorSoftStop("+dec_duration_ms+") Device_Keigan_Motor "+self.device_uuid);
            });
    };

    /**
     * モーターをハードに減速し停止させる
     */
    _AbstractDevice_Keigan_Motor.prototype.bleMotorHardStop=function(){
        UTL_LOG("bleMotorHardStop");
        var self=this;
        var buffer = new ArrayBuffer(1);
        new DataView(buffer).setUint8(0,0x81);
        BLECharacteristicController.Send(this.device_uuid, this.GATTProfile.Service, this.GATTProfile.KM_MOTOR_CONTROL,buffer,
            function(e){
                //info::モーターの状態はMeasurementNotifyで取得

            },
            function(e){
                self.CallError("bleMotorHardStop Device_Keigan_Motor "+self.device_uuid);
            });
    };
    /**
     * モーターを最大トルクでハードに停止する (ロック)
     */
    _AbstractDevice_Keigan_Motor.prototype.bleMotorHardBrake=function(){
        UTL_LOG("bleMotorHardBrake");
        var self=this;
        var buffer = new ArrayBuffer(1);
        new DataView(buffer).setUint8(0,0x82);
        BLECharacteristicController.Send(this.device_uuid, this.GATTProfile.Service, this.GATTProfile.KM_MOTOR_CONTROL,buffer,
            function(e){
                //info::モーターの状態はMeasurementNotifyで取得

            },
            function(e){
                self.CallError("bleMotorHardBrake Device_Keigan_Motor "+self.device_uuid);
            });
    };
    /**
     * モーター加減速曲線の種類を設定
     * @param val
     */
    _AbstractDevice_Keigan_Motor.prototype.bleMotorSetCurveType=function(val){
        var self=this;
        var buffer = new ArrayBuffer(6);
        new DataView(buffer).setUint8(0,self.KM_MOTOR_SETTING_COMMAND.curveType);
        new DataView(buffer).setUint16(1,0,false);//タグID 現状不使用 0
        new DataView(buffer).setUint8(3,val);
        new DataView(buffer).setUint16(4,0,false);//CRC 現状不使用 0

        BLECharacteristicController.Send(this.device_uuid, this.GATTProfile.Service, this.GATTProfile.KM_MOTOR_SETTING,buffer,
            function(e){
            },
            function(e){
                self.CallError("bleMotorSetCurveType Device_Keigan_Motor "+self.device_uuid);
            }
        );
    };
    /**
     * PIDパラメータの設定
     * @param motor_setting_command
     * @param val
     */
    _AbstractDevice_Keigan_Motor.prototype.bleMotorSetPIDVal=function(motor_setting_command,val){
        var self=this;
        switch(motor_setting_command){
            case self.KM_MOTOR_SETTING_COMMAND.acc:
            case self.KM_MOTOR_SETTING_COMMAND.dec:
            case self.KM_MOTOR_SETTING_COMMAND.speedP:
            case self.KM_MOTOR_SETTING_COMMAND.speedI:
            case self.KM_MOTOR_SETTING_COMMAND.positionP:
            case self.KM_MOTOR_SETTING_COMMAND.qCurrentP:
            case self.KM_MOTOR_SETTING_COMMAND.qCurrentI:
            case self.KM_MOTOR_SETTING_COMMAND.qCurrentD:
            //
            var buffer = new ArrayBuffer(9);
            new DataView(buffer).setUint8(0,motor_setting_command);
            new DataView(buffer).setUint16(1,0,false);//タグID 現状不使用 0
            new DataView(buffer).setFloat32(3,val,false);//ビッグエンディアン
            new DataView(buffer).setUint16(7,0,false);//CRC 現状不使用 0

            BLECharacteristicController.Send(this.device_uuid, this.GATTProfile.Service, this.GATTProfile.KM_MOTOR_SETTING,buffer,
                function(e){

                },
                function(e){
                    self.CallError("bleMotorSetPIDVal Device_Keigan_Motor "+self.device_uuid);
                }
            );
        }
    };

    /**
     * モーターのPIDパラメタを全てリセット
     */
    _AbstractDevice_Keigan_Motor.prototype.bleMotorResetPID=function(){
        var self=this;
        var buffer = new ArrayBuffer(5);
        new DataView(buffer).setUint8(0,this.KM_MOTOR_SETTING_COMMAND.resetPID);
        new DataView(buffer).setUint16(1,0, false);//タグID 現状不使用 0
        new DataView(buffer).setUint16(3,0, false);//CRC 現状不使用 0

        BLECharacteristicController.Send(this.device_uuid, this.GATTProfile.Service, this.GATTProfile.KM_MOTOR_SETTING,buffer,
            function(e){
                //デフォルトのPIDパラメータをモータから読み出し
                self.bleMotorReadRegister(self.KM_MOTOR_SETTING_COMMAND.acc);
                self.bleMotorReadRegister(self.KM_MOTOR_SETTING_COMMAND.dec);
                self.bleMotorReadRegister(self.KM_MOTOR_SETTING_COMMAND.speedP);
                self.bleMotorReadRegister(self.KM_MOTOR_SETTING_COMMAND.speedI);
                self.bleMotorReadRegister(self.KM_MOTOR_SETTING_COMMAND.positionP);
                self.bleMotorReadRegister(self.KM_MOTOR_SETTING_COMMAND.qCurrentP);
                self.bleMotorReadRegister(self.KM_MOTOR_SETTING_COMMAND.qCurrentI);
                self.bleMotorReadRegister(self.KM_MOTOR_SETTING_COMMAND.qCurrentD);
            },
            function(e){
                self.CallError("bleMotorResetPID Device_Keigan_Motor "+self.device_uuid);
            }
        );
    };
    /**
     * レジスタ値を保存する
     */
    _AbstractDevice_Keigan_Motor.prototype.bleMotorSaveAllRegisters=function(){
        var self=this;
        var buffer = new ArrayBuffer(5);
        new DataView(buffer).setUint8(0,this.KM_MOTOR_SETTING_COMMAND.saveAllRegisters);
        new DataView(buffer).setUint16(1,0, false);//タグID 現状不使用 0
        new DataView(buffer).setUint16(3,0, false);//CRC 現状不使用 0

        BLECharacteristicController.Send(this.device_uuid, this.GATTProfile.Service, this.GATTProfile.KM_MOTOR_SETTING,buffer,
            function(e){
                //書き込み完了の通知を
                self.EventHandler.fire({type: self.EventHandlerType.OnMotorSaveAllRegistersComp, target:self});
            },
            function(e){
                self.CallError("bleMotorSaveAllRegisters Device_Keigan_Motor "+self.device_uuid);
            }
        );
    };
    /**
     * 全てのレジスタ値を工場出荷状態にする
     */
    _AbstractDevice_Keigan_Motor.prototype.bleMotorResetAllRegisters=function(){
        var self=this;
        var buffer = new ArrayBuffer(5);
        new DataView(buffer).setUint8(0,this.KM_MOTOR_SETTING_COMMAND.resetAllRegisters);
        new DataView(buffer).setUint16(1,0, false);//タグID 現状不使用 0
        new DataView(buffer).setUint16(3,0, false);//CRC 現状不使用 0

        BLECharacteristicController.Send(this.device_uuid, this.GATTProfile.Service, this.GATTProfile.KM_MOTOR_SETTING,buffer,
            function(e){
               self.bleMotorReadAllRegister();//モーターのレジスター情報を取得
            },
            function(e){
                self.CallError("bleMotorResetAllRegisters Device_Keigan_Motor "+self.device_uuid);
            }
        );
    };
    /**
     * モーターの全てのレジスター情報を取得  値はnotifyで通知される
     */
    _AbstractDevice_Keigan_Motor.prototype.bleMotorReadAllRegister=function(){
        var self=this;
        //info::現状でアプリが使用しているレジスタのみ取得

        self.bleMotorReadRegister(self.KM_MOTOR_SETTING_COMMAND.curveType);
        self.bleMotorReadRegister(self.KM_MOTOR_SETTING_COMMAND.acc);
        self.bleMotorReadRegister(self.KM_MOTOR_SETTING_COMMAND.dec);
        self.bleMotorReadRegister(self.KM_MOTOR_SETTING_COMMAND.speedP);
        self.bleMotorReadRegister(self.KM_MOTOR_SETTING_COMMAND.speedI);
        self.bleMotorReadRegister(self.KM_MOTOR_SETTING_COMMAND.positionP);
        self.bleMotorReadRegister(self.KM_MOTOR_SETTING_COMMAND.qCurrentP);
        self.bleMotorReadRegister(self.KM_MOTOR_SETTING_COMMAND.qCurrentI);
        self.bleMotorReadRegister(self.KM_MOTOR_SETTING_COMMAND.qCurrentD);
    };
    /**
     * ●通知
     * 指定のレジスタ値を取得する 値はnotifyで通知される
     */
    _AbstractDevice_Keigan_Motor.prototype.bleMotorReadRegister=function(commandId){
        var self=this;
        var buffer = new ArrayBuffer(6);
        new DataView(buffer).setUint8(0,this.KM_MOTOR_SETTING_COMMAND.read);
        new DataView(buffer).setUint16(1,0, false);//タグID 現状不使用 0
        new DataView(buffer).setUint8(3,commandId);
        new DataView(buffer).setUint16(4,0, false);//CRC 現状不使用 0

        BLECharacteristicController.Send(this.device_uuid, this.GATTProfile.Service, this.GATTProfile.KM_MOTOR_SETTING,buffer,
            function(e){
            },
            function(e){
                self.CallError("bleMotorReadRegister Device_Keigan_Motor "+self.device_uuid);
            }
        );
    };
    /**
     * ●通知
     * info::モーターのレジスタ情報の取得通知の開始・停止
     */
    _AbstractDevice_Keigan_Motor.prototype.bleMotorSettingNotify=function(flg){
        var self=this;
        self.PROP.SettingNotify=flg;
        if(flg){
            BLECharacteristicController.StartNotification(this.device_uuid, this.GATTProfile.Service, this.GATTProfile.KM_MOTOR_SETTING,
                function(e){
                    if(!e){
                        return;
                    }
                    //データのparse
                    var notifyBin=new DataView(e);//6+nバイト　データ仕様　https://docs.google.com/spreadsheets/d/1uxJu86Xe8KbIlxu5oPFv9KQdvHY33-NIy0cdSgInoUk/edit#gid=1000482383
                    var notifyBinLen=notifyBin.byteLength;
                    var notifyCmd=notifyBin.getUint8(0);//レジスタ情報通知コマンドID 0x40固定

                    if(notifyCmd!=0x40||notifyBinLen<=6){return;}//レジスタ情報を含まないデータの破棄

                    var id=notifyBin.getUint16(1,false);//送信ID
                    var registerCmd=notifyBin.getUint8(3);//レジスタコマンド
                    var crc=notifyBin.getUint16(notifyBinLen-2,false);//CRCの値 最後から2dyte

                    //コマンド別によるレジスタの値の取得[4-n byte]
                    switch(registerCmd){
                        case self.KM_MOTOR_SETTING_COMMAND.curveType:
                            self._CurveType=notifyBin.getUint8(4);
                            break;
                        case self.KM_MOTOR_SETTING_COMMAND.acc:
                            self._Acc=notifyBin.getFloat32(4,false);
                            break;
                        case self.KM_MOTOR_SETTING_COMMAND.dec:
                            self._Dec=notifyBin.getFloat32(4,false);
                            break;
                        case self.KM_MOTOR_SETTING_COMMAND.speedP:
                            self._SpeedP=notifyBin.getFloat32(4,false);
                            break;
                        case self.KM_MOTOR_SETTING_COMMAND.speedI:
                            self._SpeedI=notifyBin.getFloat32(4,false);
                            break;
                        case self.KM_MOTOR_SETTING_COMMAND.positionP:
                            self._PositionP=notifyBin.getFloat32(4,false);
                            break;
                        case self.KM_MOTOR_SETTING_COMMAND.qCurrentP:
                            self._QcurrentP=notifyBin.getFloat32(4,false);
                            break;
                        case self.KM_MOTOR_SETTING_COMMAND.qCurrentI:
                            self._QcurrentI=notifyBin.getFloat32(4,false);
                            break;
                        case self.KM_MOTOR_SETTING_COMMAND.qCurrentD:
                            self._QcurrentD=notifyBin.getFloat32(4,false);
                            break;
                    }
                    self.EventHandler.fire({type: self.EventHandlerType.OnMotorRegisterValReceiveChange, target:self});
                },
                function(e){
                    self.CallError("bleMotorSettingNotify("+flg+")　Device_Keigan_Motor "+self.device_uuid);
                });
            self.bleMotorReadAllRegister();//モーターのレジスター情報を取得
        }else{
            BLECharacteristicController.StopNotification(this.device_uuid, this.GATTProfile.Service, this.GATTProfile.KM_MOTOR_SETTING,
                function(e){

                },
                function(e){
                    self.CallError("bleMotorSettingNotify("+flg+")　Device_Keigan_Motor "+self.device_uuid);
                });
        }
    };

    /**
     * ●通知
     * info::モーターの回転情報(位置・速度・トルク)測定値の取得通知の開始・停止
     */
    _AbstractDevice_Keigan_Motor.prototype.bleMotorMeasurementNotify=function(flg){
        var self=this;
        self.PROP.MeasurementNotify=flg;
        if(flg){
          //info::重複コマンド解除は不要？？  BLECharacteristicController.StopNotification(this.device_uuid, this.GATTProfile.Service, this.GATTProfile.KM_MOTOR_MEASUREMENT,null,null);
            BLECharacteristicController.StartNotification(this.device_uuid, this.GATTProfile.Service, this.GATTProfile.KM_MOTOR_MEASUREMENT,
                function(e){
                    //通知毎に毎回設定されるので、値変化時にのみ通知
                    if(self.PROP.MeasurementNotify===false){
                        self.PROP.MeasurementNotify=true;
                        self.EventHandler.fire({type: self.EventHandlerType.OnMeasurementNotifyChange, target:self});
                    }
                    if(!e){
                        return;
                    }
                    //データのparse
                    var notifyBin=new DataView(e);//12バイト　データ仕様　https://docs.google.com/spreadsheets/d/1uxJu86Xe8KbIlxu5oPFv9KQdvHY33-NIy0cdSgInoUk/edit#gid=230810792
                    var _Position=notifyBin.getFloat32(0,false);//絶対位置(radians)　浮動小数を4byte バイトオーダーはビッグエンディアン
                    var _Speed=notifyBin.getFloat32(4,false);//速度(radians / second)
                    var _Torque=notifyBin.getFloat32(8,false);//トルク(N・m)

                    //_Torque=Math.random()*0.3;//info::debug
                    ///揺らぎをフィルタリングする為、有効桁数で正規化
                    self._Position=Math.round(_Position*100)/100;//小数三位
                    //self._Position=Math.round(_Position*1000)/1000;//小数四位 //info::変速比MAX10とすると小数三位では誤差が大きい為
                    self._Speed=Math.round(_Speed*10)/10;//小数一位
                    self._Torque=Math.round(_Torque*1000)/1000;//小数三位
                    //self._IsRotationStateUpdated=true;
                    self.EventHandler.fire({type: self.EventHandlerType.OnMotorRotationStateChange, target:self}); //info::現状毎回イベントを発火してる。負荷に影響しないか検討
                },
                function(e){
                    self.CallError("bleMotorMeasurementNotify("+flg+")　Device_Keigan_Motor "+self.device_uuid);
                });
        }else{
            BLECharacteristicController.StopNotification(this.device_uuid, this.GATTProfile.Service, this.GATTProfile.KM_MOTOR_MEASUREMENT,
                function(e){
                    //通知毎に毎回設定されるので、値変化時にのみ通知
                    if(self.PROP.MeasurementNotify===true){
                        self.PROP.MeasurementNotify=false;
                        self.EventHandler.fire({type: self.EventHandlerType.OnMeasurementNotifyChange, target:self});
                    }
                },
                function(e){
                    self.CallError("bleMotorMeasurementNotify("+flg+")　Device_Keigan_Motor "+self.device_uuid);
                    //UTL_LOG("ERR:bleMotorMeasurementNotify("+flg+")　Device_Keigan_Motor "+self.device_uuid+":"+self.device_name);
                });
        }
    };





//----------------------------------------------------------------------------------------------------------------//
//  section::##### ビルトインセンサーデバイス
//----------------------------------------------------------------------------------------------------------------//





//----------------------------------------------------------------------------------------------------------------//
//  section::##### 入力コントローラー抽象クラス
//----------------------------------------------------------------------------------------------------------------//
    /************************************************************************************
     * section::***** 入力コントローラー（基底クラス）
     * ●プロパティ
     * ●イベント
     ************************************************************************************/
    function _AbstractDevice_inputctl_base(option,class_option){
        "use strict";
        var self=this;
        _AbstractDevice_base.apply(this, arguments);

        //遠隔操作用の接続登録
      //  WebRTCController.addInputCTL(this.UID,this.input_val);

        //todo::遠隔操作可能時のデバイスの有効検知（デフォルトはfalse） this.status:bool

        ///破棄時のイベント（基底クラスから呼び出される）　
        this.EventHandler.addListener(this.EventHandlerType.OnDestory,function(e){
            //廃棄時の遠隔操作用の接続解除
          //  WebRTCController.removeInputCTL(e.UID);
        });

        self.status=true;//info::入力コントローラーは全てtrue (ジャイロ等のセンサーは別途設定)
    }
    //デバイス基底クラス継承
    _AbstractDevice_inputctl_base.prototype = Object.create(_AbstractDevice_base.prototype,{constructor:{value:_AbstractDevice_inputctl_base}});
    // -------------------//
    //  パブリックプロパティ
    //--------------------//


    //  todo::ジャイロ途中
    // /************************************************************************************
    //  *  section::***** ジャイロスコープ_AbstractDevice_inputctl_gyroscope
    //  * ●プロパティ
    //  * 　 SensorWatchTime(RW):int センサーから値の取得間隔
    //  *    Active(RW):bool センサーの有効無効
    //  * ●イベント
    //  *
    //  * ●バインダー
    //  *   Binder_SensorOUT
    //  *       ○(OUT)　ST_Vector3(deg[ST_Vector3:0～360] or radian[ST_Vector3:0〜2PI] or 相対[ST_Vector3:-PI〜PI])
    //  *               絶対又は相対座標値(radian or deg)
    //  *       ○(IN)  ST_Vector3(radian[ST_Vector3:0〜2PI] or deg[ST_Vector3:0～360])
    //  *               絶対座標値(radian or deg)
    //  ************************************************************************************/
    // function _AbstractDevice_inputctl_gyroscope(option,class_option){
    //     "use strict";
    //     _AbstractDevice_inputctl_base.apply(this,arguments);
    //     var self=this;
    //     //--------------------//
    //     //追加イベント
    //     //--------------------//
    //
    //     //--------------------//
    //     //section::内部プロパティ
    //     //--------------------//
    //     this._SensorValWatchID=null;
    //     this._direction_vec3=[0,0,0];
    //     this._integ_direction_vec3=[0,0,0];//デバイスの向きのベクトル　(加速度の積算)
    //     this._ReceiveSensorCB=undefined;//ジャイロ値通知用のCB(イベント経由だと負荷が大きい)
    //     this.status=false;
    //     this._old_direction_vec3=[0,0,0];
    //     this._AxisSwapIndex=[0,1,2];
    //     // this.old_Roll_Y=0;
    //     // this.integ_Roll_Y=0;
    //
    //     //--------------------//
    //     //section::保存プロパティの復帰＆デフォルト値設定
    //     //--------------------//
    //     this.PROP.SensorWatchInterval=500;
    //     this.PROP.Active=Boolean(option.Active);
    //     this.PROP.AxisSwap=UTL_N(option.AxisSwap);
    //     //--------------------//
    //     //section::バインダーの生成
    //     //--------------------//
    //     self.Binder_SensorOUT=this.CreateBinder(
    //         {
    //             binder_name:"SensorOUT"
    //             ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_INPUT|DeviceValBinder_TYPE_LIST.VAL_VECTOR3
    //             ,allow_list:DeviceValBinder_TYPE_LIST.VAL_VECTOR3
    //             ,dataflowcb_in:null
    //             ,dataflowcb_get:null
    //             ,is_dataflow_out:true
    //             ,is_dataflow_pull:false
    //             ,option:option
    //         });
    //     self.Binder_SensorIntegOUT=this.CreateBinder(
    //         {
    //             binder_name:"SensorIntegOUT"
    //             ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_INPUT|DeviceValBinder_TYPE_LIST.VAL_VECTOR3
    //             ,allow_list:DeviceValBinder_TYPE_LIST.VAL_VECTOR3
    //             ,dataflowcb_in:null
    //             ,dataflowcb_get:null
    //             ,is_dataflow_out:true
    //             ,is_dataflow_pull:false
    //             ,option:option
    //         });
    //
    //     self.Binder_ClearIntegDirection=this.CreateBinder(
    //         {
    //             binder_name:"ClearIntegDirection"
    //             ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_INPUT|DeviceValBinder_TYPE_LIST.VAL_INT|DeviceValBinder_TYPE_LIST.VAL_STRING
    //             ,allow_list:DeviceValBinder_TYPE_LIST.VAL_INT|DeviceValBinder_TYPE_LIST.VAL_STRING
    //             ,dataflowcb_in:function(val){
    //                 self.SETClearIntegDirection=true;
    //             }
    //             ,dataflowcb_get:null
    //             ,is_dataflow_out:true
    //             ,is_dataflow_pull:false
    //             ,option:option
    //         });
    //
    //     //--------------------//
    //     //  初期化
    //     //--------------------//
    //     //デバイスの有効検知 ビルトインセンサーが無効の場合の処理
    //     this._DoAxisSwap();
    //     navigator.accelerometer.getCurrentAcceleration(function(acceleration){
    //         self.status=true;
    //         //エントリポイント
    //         self.Active=self.Active;//センサーの自動起動
    //         AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnIconViewReDrowCall,device:self});//アイコン再描画用
    //     }, function(){
    //         self.status=false;
    //         AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnIconViewReDrowCall,device:self});//アイコン再描画用
    //     });
    //
    //     //--------------------//
    //     //section::編集用UIに公開するプロパティ
    //     //--------------------//
    //     this.PublishUIProp({prop_name:"Active",prop_show_name:"Active",prop_type:PublishUIProp_TYPE.BOOL,lowlim:null,maxlim:null,step:null});
    //     this.PublishUIProp({prop_name:"SensorWatchInterval",prop_show_name:"SensorWatchInterval",prop_type:PublishUIProp_TYPE.INT,lowlim:100,maxlim:2000,step:100});
    //     this.PublishUIProp({prop_name:"SETClearIntegDirection",prop_show_name:"SETClearIntegDirection",prop_type:PublishUIProp_TYPE.SEND_NULL,lowlim:null,maxlim:null,step:null,ui_show_type:PublishUIProp_UI_SHOW_TYPE.Normal});
    //     this.PublishUIProp({prop_name:"AxisSwap",prop_show_name:"AxisSwap",prop_type:PublishUIProp_TYPE.ENUM,lowlim:null,maxlim:null,step:this.AXIS_SWAP});
    //     // -------------------//
    //     //section::イベントバインド
    //     //--------------------//
    //     ///破棄時のイベント（基底クラスから呼び出される）　
    //     this.EventHandler.addListener(this.EventHandlerType.OnDestory,function(e){
    //         self._SetSensorActive(false);//停止
    //     });
    //
    // }
    // //基底クラス継承
    // _AbstractDevice_inputctl_gyroscope.prototype = Object.create(_AbstractDevice_inputctl_base.prototype,{constructor:{value:_AbstractDevice_inputctl_gyroscope}});
    // // -------------------//
    // // section::定数定義
    // //--------------------//
    // //回転軸の出力順　RevolvingAxisSwap
    // _AbstractDevice_inputctl_gyroscope.prototype.AXIS_SWAP={
    //     "[X,Y,Z]":0,
    //     "[X,Z,Y]":1,
    //     "[Y,X,Z]":2,
    //     "[Y,Z,X]":3,
    //     "[Z,X,Y]":4,
    //     "[Z,Y,X]":5,
    // };
    // //--------------------//
    // //  パブリックプロパティ
    // //--------------------//
    // //SensorWatchInterval 加速度情報の取得間隔ms
    // Object.defineProperty(_AbstractDevice_inputctl_gyroscope.prototype, 'SensorWatchInterval', {
    //     get: function() {
    //         return this.PROP.SensorWatchInterval;
    //     },
    //     set:function(val){
    //         this.PROP.SensorWatchInterval = val?val:500;
    //         this._SetSensorActive(this.PROP.Active,this.PROP.SensorWatchInterval);//取得タイミングをリセット
    //     }
    // });
    // //方向の積算値(X,Y,Z)
    // Object.defineProperty(_AbstractDevice_inputctl_gyroscope.prototype, 'Direction', {
    //     get: function() {
    //         return this._direction_vec3.concat();
    //     },
    //     set:function(val){
    //     }
    // });
    // //方向の積算値のクリアー
    // Object.defineProperty(_AbstractDevice_inputctl_gyroscope.prototype, 'SETClearIntegDirection', {
    //     get: function() {
    //         return null;
    //     },
    //     set:function(val){
    //         this._clear_integ_direction();
    //     }
    // });
    //
    // //Active(RW):bool センサーの有効無効
    // Object.defineProperty(_AbstractDevice_inputctl_gyroscope.prototype, 'Active', {
    //     get: function() {
    //         return this.PROP.Active;
    //     },
    //     set:function(flg){
    //         this._SetSensorActive(flg,this.PROP.SensorWatchInterval);
    //     }
    // });
    // //AxisSwap(RW):Enum 出力軸の入れ替え
    // Object.defineProperty(_AbstractDevice_inputctl_gyroscope.prototype, 'AxisSwap', {
    //     get: function() {
    //         return this.PROP.AxisSwap;
    //     },
    //     set:function(val){
    //         this.PROP.AxisSwap=val;
    //         this._DoAxisSwap();
    //     }
    // });
    //
    //
    // //--------------------//
    // //  プライベートメソッド
    // //--------------------//
    // /// 加速度情報の取得を開始/停止　flg:bool 開始:true  time:int 取得する間隔ms
    // _AbstractDevice_inputctl_gyroscope.prototype._SetSensorActive=function(StartFlg,time){
    //     var self=this;
    //     if(!self.status){
    //         self.PROP.Active=false;
    //         return;
    //     }
    //
    //     self.PROP.Active=StartFlg;
    //     if(self._SensorValWatchID){
    //         navigator.accelerometer.clearWatch(self._SensorValWatchID);
    //         self._SensorValWatchID=null;
    //     }
    //     if(self.PROP.Active){
    //         self._SensorValWatchID= navigator.accelerometer.watchAcceleration(
    //             function(acceleration){
    //                 //todo::ロジック検討　一端後回し
    //                 //
    //
    //                 self._direction_vec3[0]=acceleration.x*0.1; //値を-1〜1に正規化
    //                 self._direction_vec3[1]=acceleration.y*0.1;
    //                 self._direction_vec3[2]=acceleration.z*0.1;
    //                 self._SensorValPush();
    //
    //
    //                 return;//todo::debug
    //
    //
    //                 // self._integ_direction_vec3[0]+=acceleration.x;//*0.1;
    //                 // self._integ_direction_vec3[1]+=acceleration.y;//*0.1;
    //                 // self._integ_direction_vec3[2]+=acceleration.z;//*0.1;
    //
    //
    //
    //                 var n_y=Math.atan2(acceleration.y, acceleration.z) * 180/Math.PI;
    //
    //                // var n_z=Math.atan2(acceleration.x, acceleration.z) * 180/Math.PI;
    //                 var n_z=Math.atan2(-acceleration.x, Math.sqrt(acceleration.y*acceleration.y +  acceleration.z* acceleration.z)) * 180/Math.PI;
    //                 var p_y=n_y-self._old_direction_vec3[1];
    //                 var p_z=n_z-self._old_direction_vec3[2];
    //
    //                 var p_y=p_y>180?p_y-360:(p_y<-180?p_y+360:p_y);
    //                 var p_z=p_z>180?p_z-360:(p_z<-180?p_z+360:p_z);
    //
    //                 //var Pitch=Math.atan2(-acceleration.z, Math.sqrt(acceleration.y*acceleration.y +  acceleration.x* acceleration.x)) * 180/Math.PI;
    //
    //                 self._integ_direction_vec3[1]+=p_y;
    //                 self._integ_direction_vec3[2]+=p_z;
    //                 console.dir({
    //                     R_X:acceleration.x,
    //                     R_Y:acceleration.y,
    //                     R_Z:acceleration.z,
    //
    //                     Y:Math.atan2(acceleration.y, acceleration.z),
    //                     X:Math.atan2(acceleration.x, acceleration.z)
    //
    //                 });
    //                 self._old_direction_vec3[1]=n_y;
    //                 self._old_direction_vec3[2]=n_z;
    //
    //                //  var Rx=acceleration.x;
    //                //  var Ry=acceleration.y;
    //                //  var Rz=acceleration.z;
    //                //  var R =  Math.sqrt(Math.pow(Rx,2)+Math.pow(Ry,2)+Math.pow(Rz,2));
    //                //
    //                //  var  Arx = Math.acos(Rx/R)*180/Math.PI;
    //                //  var  Ary = Math.acos(Ry/R)*180/Math.PI;
    //                //  var  Arz = Math.acos(Rz/R)*180/Math.PI;
    //                // // console.dir([Arx,Ary,Arz]);
    //
    //                 // self._SensorValPush();
    //
    //             },
    //             function(){
    //                 self.PROP.Active=false;
    //                 //self.CallError("SetSensorWatch _AbstractDevice_inputctl_gyroscope");
    //                 UTL_LOG("ERR:SetSensorWatch _AbstractDevice_inputctl_gyroscope "+self.device_name);
    //             },
    //             {
    //                 frequency : time?time:500
    //             }
    //         );
    //     }
    // };
    // //加速度の向きのリセット
    // _AbstractDevice_inputctl_gyroscope.prototype._clear_integ_direction=function(){
    //     this._integ_direction_vec3[0]=0;
    //     this._integ_direction_vec3[1]=0;
    //     this._integ_direction_vec3[2]=0;
    // };
    //
    // _AbstractDevice_inputctl_gyroscope.prototype._SensorValPush=function(){
    //
    //
    //     if(this.Binder_SensorOUT.is_bind){
    //         var ps=new ST_Vector3(this._direction_vec3[this._AxisSwapIndex[0]],this._direction_vec3[this._AxisSwapIndex[1]],this._direction_vec3[this._AxisSwapIndex[2]]);
    //         this.Binder_SensorOUT.Par_Val=ps;
    //     }
    //
    //     if(this.Binder_SensorIntegOUT.is_bind){
    //         var integ_ps=new ST_Vector3(this._integ_direction_vec3[this._AxisSwapIndex[0]],this._integ_direction_vec3[this._AxisSwapIndex[1]],this._integ_direction_vec3[this._AxisSwapIndex[2]]);
    //         this.Binder_SensorIntegOUT.Par_Val=integ_ps;
    //     }
    // };
    // _AbstractDevice_inputctl_gyroscope.prototype._DoAxisSwap=function(){
    // switch(this.PROP.AxisSwap){
    //     case this.AXIS_SWAP["[X,Y,Z]"]:
    //         this._AxisSwapIndex=[0,1,2];
    //         break;
    //     case this.AXIS_SWAP["[X,Z,Y]"]:
    //         this._AxisSwapIndex=[0,2,1];
    //         break;
    //     case this.AXIS_SWAP["[Y,X,Z]"]:
    //         this._AxisSwapIndex=[1,0,2];
    //         break;
    //     case this.AXIS_SWAP["[Y,Z,X]"]:
    //         this._AxisSwapIndex=[1,2,0];
    //         break;
    //     case this.AXIS_SWAP["[Z,X,Y]"]:
    //         this._AxisSwapIndex=[2,0,1];
    //         break;
    //     case this.AXIS_SWAP["[Z,Y,X]"]:
    //         this._AxisSwapIndex=[2,1,0];
    //         break;
    //     }
    // };

    /************************************************************************************
     *  section::***** inputctl_XYPad XYトラックパッド
         スタート位置からのドラッグ方向・距離からXY [X(-1.0〜+1.0),Y(-1.0〜+1.0)]までを相対的に出力
         ●バインダー
             ◎DragVector	(表示名:DragVector)	//・ドラッグ方向のベクトル出力 {X:±1.0,Y:±1.0}
             ◎DragXYpoint『Advance』	(表示名:DragXYpoint)	//・ドラッグした座標
         ●プロパティ
             ◎ UIInput_watch_time『Advance』		//・ドラッグ中の入力値の取得間隔(ms) int[0 - 200]　負荷防止用
             ◎ min_move_dist		//・最小移動距離の閾値とする半径(px)  (この距離以上動いたらベクトルの0点として感知する)
             ◎ max_move_dist		//・最大移動距離の閾値とする半径(px) (この距離以上は1とするリミット)
             ◎ UIInput_mode		//・入力モード（相対的・中央固定）
             ◎ SwapXYPosition		//・XYの出力座標入れ替え
     ************************************************************************************/
    
    function _AbstractDevice_inputctl_XYPad(option,class_option) {
        "use strict";
        _AbstractDevice_inputctl_base.apply(this, arguments);
        var self = this;
        //--------------------//
        //section::追加イベントハンドラ定義
        //--------------------//
        //--------------------//
        //section::内部プロパティ
        //--------------------//
        this._drag_vector=new ST_Vector2();
        this._xy_vector=new ST_Vector2();
        //--------------------//
        //section::保存プロパティの復帰＆デフォルト値設定
        //--------------------//
        this.PROP.UIInput_mode=option.UIInput_mode?option.UIInput_mode:this.UI_INPUT_MODE.RELATIVE;
        this.PROP.UIInput_watch_time=option.UIInput_watch_time?option.UIInput_watch_time:100;
        this.PROP.min_move_dist=option.min_move_dist?option.min_move_dist:20;
        this.PROP.max_move_dist=option.max_move_dist?option.max_move_dist:150;
        this.PROP.SwapXYPosition=option.SwapXYPosition!=null?option.SwapXYPosition:false;
        //--------------------//
        //section::バインダーの生成
        //--------------------//
        //ドラッグ方向ベクトル
        self.Binder_DragVector=this.CreateBinder(
        {
            binder_name:"DragVector"
            ,binder_show_name:"Drag Vector"
            ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_INPUT|DeviceValBinder_TYPE_LIST.VAL_VECTOR2
            ,allow_list:DeviceValBinder_TYPE_LIST.VAL_VECTOR2
            ,dataflowcb_in:null
            ,dataflowcb_get:function(){return self._drag_vector.Clone();}
            ,is_dataflow_out:true
            ,is_dataflow_pull:false
            ,ui_show_type:DeviceValBinder_UI_SHOW_TYPE.Normal
            ,option:option
        });
        //XY座標
        self.Binder_DragXYpoint=this.CreateBinder({
            binder_name:"DragXYpoint"
            ,binder_show_name:"Drag XY Point"
            ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_INPUT|DeviceValBinder_TYPE_LIST.VAL_VECTOR2
            ,allow_list:DeviceValBinder_TYPE_LIST.VAL_VECTOR2
            ,dataflowcb_in:null
            ,dataflowcb_get:function(){return self._xy_vector.Clone();}
            ,is_dataflow_out:true
            ,is_dataflow_pull:false
            ,ui_show_type:DeviceValBinder_UI_SHOW_TYPE.Advance
            ,option:option
        });

        //--------------------//
        //section::初期化
        //--------------------//
        //--------------------//
        //section::編集用UIに公開するプロパティ
        //--------------------//
        this.PublishUIProp({prop_name:"UIInput_mode",prop_show_name:"Input Mode",prop_type:PublishUIProp_TYPE.ENUM,lowlim:null,maxlim:null,step:this.UI_INPUT_MODE,ui_show_type:PublishUIProp_UI_SHOW_TYPE.Normal});
        this.PublishUIProp({prop_name:"UIInput_watch_time",prop_show_name:"Input Interval",prop_type:PublishUIProp_TYPE.INT,lowlim:10,maxlim:1000,step:10,ui_show_type:PublishUIProp_UI_SHOW_TYPE.Advance});
        this.PublishUIProp({prop_name:"min_move_dist",prop_show_name:"Min Movement",prop_type:PublishUIProp_TYPE.INT,lowlim:10,maxlim:100,step:10,ui_show_type:PublishUIProp_UI_SHOW_TYPE.Normal});
        this.PublishUIProp({prop_name:"max_move_dist",prop_show_name:"Max Movement",prop_type:PublishUIProp_TYPE.INT,lowlim:10,maxlim:250,step:10,ui_show_type:PublishUIProp_UI_SHOW_TYPE.Normal});
        this.PublishUIProp({prop_name:"SwapXYPosition",prop_show_name:"Exchange XY",prop_type:PublishUIProp_TYPE.BOOL,lowlim:null,maxlim:null,step:null,ui_show_type:PublishUIProp_UI_SHOW_TYPE.Normal});

        // -------------------//
        //section::イベントバインド
        //--------------------//
        ///破棄時のイベント（基底クラスから呼び出される）　
        this.EventHandler.addListener(self.EventHandlerType.OnDestory,function(e){

        });
        ////
    };
    //基底クラス継承
    _AbstractDevice_inputctl_XYPad.prototype = Object.create(_AbstractDevice_inputctl_base.prototype,{constructor:{value:_AbstractDevice_inputctl_XYPad}});
    // -------------------//
    // section::定数定義
    //--------------------//
    //UI入力モード　相対 中央固定（テレプレ用）
    _AbstractDevice_inputctl_XYPad.prototype.UI_INPUT_MODE={
        RELATIVE:0,
        CENTER_FIXED:1
    };
    // -------------------//
    // section::パブリックプロパティ
    //--------------------//
    //UIドラッグ中の入力から取得する値の取得間隔
    Object.defineProperty(_AbstractDevice_inputctl_XYPad.prototype,'UIInput_watch_time', {
        get: function() {
            return this.PROP.UIInput_watch_time;
        },
        set:function(val){
            if(this.PROP.UIInput_watch_time===val){return;}
            this.PROP.UIInput_watch_time=val;
        }
    });
    //最小移動距離
    Object.defineProperty(_AbstractDevice_inputctl_XYPad.prototype,'min_move_dist', {
        get: function() {
            return this.PROP.min_move_dist;
        },
        set:function(val){
            this.PROP.min_move_dist=val;
        }
    });
    //最大移動距離
    Object.defineProperty(_AbstractDevice_inputctl_XYPad.prototype,'max_move_dist', {
        get: function() {
            return this.PROP.max_move_dist;
        },
        set:function(val){
            this.PROP.max_move_dist=val;
        }
    });
    //UI入力モード
    Object.defineProperty(_AbstractDevice_inputctl_XYPad.prototype,'UIInput_mode', {
        get: function() {
            return this.PROP.UIInput_mode;
        },
        set:function(val){
            this.PROP.UIInput_mode=val;
        }
    });
    //XY座標入れ替え
    Object.defineProperty(_AbstractDevice_inputctl_XYPad.prototype,'SwapXYPosition', {
        get: function() {
            return this.PROP.SwapXYPosition;
        },
        set:function(val){
            this.PROP.SwapXYPosition=val;
        }
    });


    // -------------------//
    // section::内部関数
    //--------------------//
    /**
     * ドラッグされたXYベクトル>バインダー送信
     */
    _AbstractDevice_inputctl_XYPad.prototype._SetPoint=function(vector2,xy_vector2){
        this._drag_vector=vector2;
        this._xy_vector=xy_vector2;
        var dv=this._drag_vector.Clone();
        var dxy=this._xy_vector.Clone();
        //XY軸の入れ替えオプション
        if(this.SwapXYPosition){
            var tmp=dv.x;
            dv.x=dv.y;
            dv.y=tmp;
            tmp=dxy.x
            dxy.x=dxy.y;
            dxy.y=tmp;
        }
        this.Binder_DragVector.Par_Val=dv; //バインダー送信
        this.Binder_DragXYpoint.Par_Val=dxy; //バインダー送信
    };

    /************************************************************************************
     * section::***** inputctl_PushBtn PUSHボタン
         ●バインダー
            ◎BtnVal	(表示名:ButtonPushVal)	//・押下時に送信するボタンの値
         ●プロパティ
             ◎ ValueType		//・ボタンのデータ型
             ◎ BtnVal		//・ボタンの値
             ◎ BtnText		//・ボタン表示名
     ************************************************************************************/
    function _AbstractDevice_inputctl_PushBtn(option,class_option) {
        "use strict";
        _AbstractDevice_inputctl_base.apply(this, arguments);
        var self = this;
        //--------------------//
        //section::追加イベントハンドラ定義
        //--------------------//
        this.EventHandlerType["OnValChange"]="OnValChange";
        this.EventHandlerType["OnBtnTextChange"]="OnBtnTextChange";//UI再描画用
        //--------------------//
        //section::内部プロパティ
        //--------------------//
        //--------------------//
        //section::保存プロパティの復帰＆デフォルト値設定
        //--------------------//
        this.PROP.ValueType=UTL_N(option.ValueType);
        this.PROP.BtnVal=option.BtnVal?option.BtnVal:0;
        this.PROP.BtnText=option.BtnText?option.BtnText:'PUSH';
        //--------------------//
        //section::バインダーの生成
        //--------------------//
        self.Binder_BtnVal=this.CreateBinder(
            {
                binder_name:"BtnVal"
                ,binder_show_name:"Button Value"
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_INPUT|DeviceValBinder_TYPE_LIST.VAL_NUMBER
                ,allow_list:DeviceValBinder_TYPE_LIST.VAL_NUMBER|DeviceValBinder_TYPE_LIST.VAL_STRING
                ,dataflowcb_in:null
                ,dataflowcb_get:function(){return self.BtnVal;}
                ,is_dataflow_out:true
                ,is_dataflow_pull:false
                ,option:option
            }
        );
        self.Binder_BtnValWrite=this.CreateBinder(
            {
                binder_name:"BtnValWrite"
                ,binder_show_name:"Button Value Write"
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_INPUT|DeviceValBinder_TYPE_LIST.VAL_NUMBER
                ,allow_list:DeviceValBinder_TYPE_LIST.VAL_NUMBER|DeviceValBinder_TYPE_LIST.VAL_STRING
                ,dataflowcb_in:function(val){self.BtnVal=val;}
                ,dataflowcb_get:function(){return self.BtnVal;}
                ,is_dataflow_out:false
                ,is_dataflow_pull:false
                ,option:option
            }
        );
        //--------------------//
        //section::編集用UIに公開するプロパティ
        //--------------------//
        this.PublishUIProp({prop_name:"ValueType",prop_show_name:"Value Type",prop_type:PublishUIProp_TYPE.ENUM,lowlim:null,maxlim:null,step:this.VALUE_TYPE});
        this.PublishUIProp({prop_name:"BtnVal",prop_show_name:"Button Value",prop_type:PublishUIProp_TYPE.STRING,lowlim:null,maxlim:null,step:null});
        this.PublishUIProp({prop_name:"BtnText",prop_show_name:"Button Text",prop_type:PublishUIProp_TYPE.STRING,lowlim:null,maxlim:null,step:null});
        //--------------------//
        //section::初期化
        //--------------------//
        this._ChangeType_Binder_val();
        // -------------------//
        //section::イベントバインド
        //--------------------//
        ///破棄時のイベント（基底クラスから呼び出される）　
        this.EventHandler.addListener(self.EventHandlerType.OnDestory,function(e){ });
    }
    //基底クラス継承
    _AbstractDevice_inputctl_PushBtn.prototype = Object.create(_AbstractDevice_inputctl_base.prototype,{constructor:{value:_AbstractDevice_inputctl_PushBtn}});
    // -------------------//
    // section::定数定義
    //--------------------//
    //作成する構造体の種類
    _AbstractDevice_inputctl_PushBtn.prototype.VALUE_TYPE={
        "NUMBER":0,"STRING":1
    };
    // -------------------//
    // section::パブリックプロパティ
    //--------------------//
    Object.defineProperty(_AbstractDevice_inputctl_PushBtn.prototype,'ValueType', {
        get: function() {
            return this.PROP.ValueType;
        },
        set:function(val){
            this.PROP.ValueType=UTL_N(val);
            this._ChangeType_Binder_val();
        }
    });
    Object.defineProperty(_AbstractDevice_inputctl_PushBtn.prototype,'BtnVal', {
        get: function() {return this.PROP.BtnVal;},
        set:function(val){
            this.PROP.BtnVal=val;
            this.EventHandler.fire({type:this.EventHandlerType.OnValChange, BtnVal:val});
        }
    });
    Object.defineProperty(_AbstractDevice_inputctl_PushBtn.prototype,'BtnText', {
        get: function() {
            return this.PROP.BtnText;
        },
        set:function(val){this.PROP.BtnText=val;
            this.EventHandler.fire({type:this.EventHandlerType.OnBtnTextChange,BtnText:val});//UI再描画用
            AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnIconViewReDrowCall,device:this});//アイコン再描画用
        }
    });
    // -------------------//
    // section::内部関数
    //--------------------//
    //ボタンの押下
    _AbstractDevice_inputctl_PushBtn.prototype._SendBtnPush=function(){
        switch (this.ValueType){
            case this.VALUE_TYPE.NUMBER:
                this.Binder_BtnVal.Par_Val=UTL_N(this.BtnVal);
                break;
            case this.VALUE_TYPE.STRING:
                this.Binder_BtnVal.Par_Val=String(this.BtnVal);
                break;
        }

    };
    //データ型 変更時のBind_OUTのバインダー型の検証と変更
    _AbstractDevice_inputctl_PushBtn.prototype._ChangeType_Binder_val=function(){
        switch (this.ValueType){
            case this.VALUE_TYPE.NUMBER:
                this.Binder_BtnVal._type_list=DeviceValBinder_TYPE_LIST.DEVICE_INPUT|DeviceValBinder_TYPE_LIST.VAL_NUMBER;
                this.Binder_BtnValWrite._type_list=DeviceValBinder_TYPE_LIST.DEVICE_INPUT|DeviceValBinder_TYPE_LIST.VAL_NUMBER;
                break;
            case this.VALUE_TYPE.STRING:
                this.Binder_BtnVal._type_list=DeviceValBinder_TYPE_LIST.DEVICE_INPUT|DeviceValBinder_TYPE_LIST.VAL_STRING;
                this.Binder_BtnValWrite._type_list=DeviceValBinder_TYPE_LIST.DEVICE_INPUT|DeviceValBinder_TYPE_LIST.VAL_STRING;
                break;
        }
        //適合しない場合は強制的に解除
        if(!this.Binder_BtnVal.is_binding_compatible){
            this.Binder_BtnVal.UnBinding();
            AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnBindLineViewReDrowCall,device:this});//バインド線再描画用
        }
        if(!this.Binder_BtnValWrite.is_binding_compatible){
            this.Binder_BtnValWrite.UnBinding();
            AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnBindLineViewReDrowCall,device:this});//バインド線再描画用
        }
    };
    /************************************************************************************
     * section::***** inputctl_NumberSlider 数値生成スライダー
         ●バインダー
            ◎SliderVal	(表示名:SliderVal)	//・スライダーの値
         ●プロパティ
             ◎ SliderVal 『Advance』		//・スライダーの現在の値
             ◎ Low		//・スライダーの下限値
             ◎ Max		//・スライダーの上限値
             ◎ Step		//・スライダーのステップ値
             ◎ RealtimeChange		//・スライド中もリアルタイムで値を変更するか
             ◎ FixMidpoint		//・離した時に自動で中央に戻る
             ◎ ValueInvert		//・スライダー左右の値を入れ替える
     ************************************************************************************/
    function _AbstractDevice_inputctl_NumberSlider(option,class_option) {
        "use strict";
        _AbstractDevice_inputctl_base.apply(this, arguments);
        var self = this;
        //--------------------//
        //section::追加イベントハンドラ定義
        //--------------------//
        self.EventHandlerType["OnValChange"]="OnValChange";
        self.EventHandlerType["OnLowChange"]="OnLowChange";
        self.EventHandlerType["OnMaxChange"]="OnMaxChange";
        self.EventHandlerType["OnStepChange"]="OnStepChange";
        self.EventHandlerType["OnRealtimeChange"]="OnRealtimeChange";
        self.EventHandlerType["OnFixMidpointChange"]="OnFixMidpointChange";
        self.EventHandlerType["OnValueInvertChange"]="OnValueInvertChange";
        //--------------------//
        //section::内部プロパティ
        //--------------------//
        //--------------------//
        //section::保存プロパティの復帰＆デフォルト値設定
        //--------------------//
        self.PROP.SliderVal=option.SliderVal?option.SliderVal:0;
        self.PROP.Low=option.Low?option.Low:0;
        self.PROP.Max=option.Max?option.Max:100;
        self.PROP.Step=option.Step?option.Step:1;
        self.PROP.RealtimeChange=option.RealtimeChange===undefined?false:(option.RealtimeChange?true:false);
        self.PROP.FixMidpoint=option.FixMidpoint===undefined?false:(option.FixMidpoint?true:false);
        self.PROP.ValueInvert=option.ValueInvert===undefined?false:(option.ValueInvert?true:false);
        //--------------------//
        //section::バインダーの生成
        //--------------------//
        self.Binder_val=this.CreateBinder(
            {
                binder_name:"SliderVal"
                ,binder_show_name:'Slider Value'
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_INPUT|DeviceValBinder_TYPE_LIST.VAL_FLOAT
                ,allow_list:DeviceValBinder_TYPE_LIST.VAL_INT | DeviceValBinder_TYPE_LIST.VAL_FLOAT
                ,dataflowcb_in:null
                ,dataflowcb_get:function(){return self.SliderVal;}
                ,is_dataflow_out:true
                ,is_dataflow_pull:false
                ,option:option
            }
        );
        self.Binder_valWrite=this.CreateBinder(
            {
                binder_name:"SliderValWrite"
                ,binder_show_name:'Slider Value Write'
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_INPUT|DeviceValBinder_TYPE_LIST.VAL_FLOAT
                ,allow_list:DeviceValBinder_TYPE_LIST.VAL_INT | DeviceValBinder_TYPE_LIST.VAL_FLOAT
                ,dataflowcb_in:function(val){self.SliderVal=val;}
                ,dataflowcb_get:function(){return self.SliderVal;}
                ,is_dataflow_out:false
                ,is_dataflow_pull:false
                ,option:option
            }
        );

        //--------------------//
        //section::編集用UIに公開するプロパティ
        //--------------------//
        self.PublishUIProp({prop_name:"SliderVal",prop_show_name:"Slider Value",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:null,maxlim:null,step:null,ui_show_type:PublishUIProp_UI_SHOW_TYPE.Advance});
        self.PublishUIProp({prop_name:"Low",prop_show_name:"Min",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:null,maxlim:null,step:null});
        self.PublishUIProp({prop_name:"Max",prop_show_name:"Max",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:null,maxlim:null,step:null});
        self.PublishUIProp({prop_name:"Step",prop_show_name:"Step Value",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:null,maxlim:null,step:null});
        self.PublishUIProp({prop_name:"RealtimeChange",prop_show_name:"Input in Real-time",prop_type:PublishUIProp_TYPE.BOOL,lowlim:null,maxlim:null,step:null,ui_show_type:PublishUIProp_UI_SHOW_TYPE.Normal});
        self.PublishUIProp({prop_name:"FixMidpoint",prop_show_name:"Auto Return to Mid",prop_type:PublishUIProp_TYPE.BOOL,lowlim:null,maxlim:null,step:null});
        self.PublishUIProp({prop_name:"ValueInvert",prop_show_name:"Value Invert",prop_type:PublishUIProp_TYPE.BOOL,lowlim:null,maxlim:null,step:null});
        //--------------------//
        //section::初期化
        //--------------------//
        // -------------------//
        //section::イベントバインド
        //--------------------//
        ///破棄時のイベント（基底クラスから呼び出される）　
        self.EventHandler.addListener(self.EventHandlerType.OnDestory,function(e){ });
    }
    //基底クラス継承
    _AbstractDevice_inputctl_NumberSlider.prototype = Object.create(_AbstractDevice_inputctl_base.prototype,{constructor:{value:_AbstractDevice_inputctl_NumberSlider}});
    // -------------------//
    // section::定数定義
    //--------------------//
    // -------------------//
    // section::パブリックプロパティ
    //--------------------//

    Object.defineProperty(_AbstractDevice_inputctl_NumberSlider.prototype,'Val', {
        get: function() {return this.PROP.SliderVal;},
        set:function(val){
            this.PROP.SliderVal=val;
            this.EventHandler.fire({type:this.EventHandlerType.OnValChange, val:val});
        }
    });
    Object.defineProperty(_AbstractDevice_inputctl_NumberSlider.prototype,'Low', {
        get: function() {return this.PROP.Low;},
        set:function(val){
            this.PROP.Low=val;
            this.EventHandler.fire({type:this.EventHandlerType.OnLowChange, val:val});
        }
    });
    Object.defineProperty(_AbstractDevice_inputctl_NumberSlider.prototype,'Max', {
        get: function() {return this.PROP.Max;},
        set:function(val){
            this.PROP.Max=val;
            this.EventHandler.fire({type:this.EventHandlerType.OnMaxChange, val:val});
        }
    });
    Object.defineProperty(_AbstractDevice_inputctl_NumberSlider.prototype,'Step', {
        get: function() {return this.PROP.Step;},
        set:function(val){
            this.PROP.Step=val;
            this.EventHandler.fire({type:this.EventHandlerType.OnStepChange, val:val});
        }
    });
    Object.defineProperty(_AbstractDevice_inputctl_NumberSlider.prototype,'RealtimeChange', {
        get: function() {return this.PROP.RealtimeChange;},
        set:function(val){
            this.PROP.RealtimeChange=val;
           this.EventHandler.fire({type:this.EventHandlerType.OnRealtimeChange, val:val});
        }
    });
    Object.defineProperty(_AbstractDevice_inputctl_NumberSlider.prototype,'FixMidpoint', {
        get: function() {return this.PROP.FixMidpoint;},
        set:function(val){
            this.PROP.FixMidpoint=val;
            if(val){
                //中央値の生成
                var c_val=this.Low+(this.Max-this.Low)/2;
                this.SliderVal=c_val;
            }
            this.EventHandler.fire({type:this.EventHandlerType.OnFixMidpointChange, val:val});
        }
    });
    Object.defineProperty(_AbstractDevice_inputctl_NumberSlider.prototype,'ValueInvert', {
        get: function() {return this.PROP.ValueInvert;},
        set:function(val){
            this.PROP.ValueInvert=val;
            this.EventHandler.fire({type:this.EventHandlerType.OnValueInvertChange, val:val});
        }
    });
    // -------------------//
    // section::内部関数
    //--------------------//
    _AbstractDevice_inputctl_NumberSlider.prototype._SetVal=function(val){
        this.PROP.SliderVal=val;
        this.Binder_val.Par_Val= val;//バインダー送信
    };
    /************************************************************************************
     * section::***** inputctl_MTControlKit モーターコントロールセット(学習用)
         ●バインダー
            ◎Motor	(表示名:Motor)	//・モーターの制御プロトコル(モーターに接続)
         ●プロパティ 無し
     ************************************************************************************/
    function _AbstractDevice_inputctl_MTControlKit(option,class_option) {
        "use strict";
        _AbstractDevice_inputctl_base.apply(this, arguments);
        var self = this;
        //--------------------//
        //section::追加イベントハンドラ定義
        //--------------------//
        this.EventHandlerType["OnModePlayChange"]="OnModePlayChange";
        this.EventHandlerType["OnMTRotStateChange"]="OnMTRotStateChange";
        this.EventHandlerType["OnPossionMemorysChange"]="OnPossionMemorysChange";

        //--------------------//
        //section::内部プロパティ
        //--------------------//
        this._Rev_MTRotState={};//モーターの状態X notifyに連動
        this._Rev_LedStateColor={};//new ST_MTLedState();//LEDの色
        //this._PreSetPositon_Motor={};
        this._PossionMemorys=[];//保存位置
        this._Speed=10;//rpm
        this._PosSpeed=10;//rpm
        this._PosTorque=1;
        this._PosPosition=0;
        this._IsSpeedModePlay=false;
        this._IsPositionModePlay=false;
        //--------------------//
        //section::保存プロパティの復帰＆デフォルト値設定
        //--------------------//

        //--------------------//
        //section::バインダーの生成
        //--------------------//
        self.Binder_Motor=this.CreateBinder(
            {
                binder_name:"Motor"
                ,binder_show_name:"Motor"
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_INPUT | DeviceValBinder_TYPE_LIST.VAL_CTL_MT
                ,allow_list:DeviceValBinder_TYPE_LIST.DEVICE_MOTOR | DeviceValBinder_TYPE_LIST.VAL_CTL_MT
                ,dataflowcb_in:function(val){
                if(val instanceof ST_NotifyProtocol){
                    if(val.Val instanceof ST_MTRotState){
                        // //回転情報受信
                        // if(!self._Rev_MTRotState[val.originID]){
                        //     self._PreSetPositon_Motor[val.originID]=val.Val.Position;
                        // }
                        self._Rev_MTRotState[val.originID]=val.Val;
                        self.EventHandler.fire({type:self.EventHandlerType.OnMTRotStateChange,device:self});//PLAY画面 回転情報表示更新用
                    }else if(val.Val instanceof ST_MTLedState){
                        //LED情報受信
                        self._Rev_LedStateColor[val.originID]=val.Val;
                        //AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnIconViewLedStateReDrowCall,device:self});//LEDアイコン再描画用
                        AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnIconViewReDrowCall,device:self});//アイコン再描画用
                    }
                }
            }
                ,dataflowcb_get:null
                ,is_dataflow_out:true
                ,is_dataflow_pull:false
                ,option:option
            }
        );

        //--------------------//
        //section::編集用UIに公開するプロパティ
        //--------------------//

        //--------------------//
        //section::初期化
        //--------------------//
        // -------------------//
        //section::イベントバインド
        //--------------------//
        self.Binder_Motor.EventHandler.addListener(self.Binder_Motor.EventHandlerType.OnBindChange,function(e){
            //LED情報の取得要求
            self.Binder_Motor.Par_Val=new ST_CtlProtocol("GetLedStateColorNotifyOne",true);
        });
        ///破棄時のイベント（基底クラスから呼び出される）　
        self.EventHandler.addListener(self.EventHandlerType.OnDestory,function(e){ });
    }
    //基底クラス継承
    _AbstractDevice_inputctl_MTControlKit.prototype = Object.create(_AbstractDevice_inputctl_base.prototype,{constructor:{value:_AbstractDevice_inputctl_MTControlKit}});
    // -------------------//
    // section::定数定義
    //--------------------//
    // -------------------//
    // section::パブリックプロパティ
    //--------------------//


    // -------------------//
    // section::内部関数
    //--------------------//
    _AbstractDevice_inputctl_MTControlKit.prototype._SetResetPosition=function(){
        this.Binder_Motor.Par_Val=new ST_CtlProtocol("PreSetPositon",0);
        this._PossionMemorys=[];//保存位置の全消去
        this.EventHandler.fire({type:this.EventHandlerType.OnPossionMemorysChange,val:this});//UI再描画用
    };
    //位置の記録処理　num:int[0 - n] 記憶位置を保存するメモリ番号
    _AbstractDevice_inputctl_MTControlKit.prototype._SetRecPosition=function(num){
        var pos={};
        var key=Object.keys(this._Rev_MTRotState);
        for(var i=0;i<key.length;i++) {
            var k=key[i];
            //pos[k]=UTL_N(this._Rev_MTRotState[k].Position-this._PreSetPositon_Motor[k]);
            pos[k]=UTL_N(this._Rev_MTRotState[k].Position);
        }
        this._PossionMemorys[UTL_INT(num)]=pos;
        this.EventHandler.fire({type:this.EventHandlerType.OnPossionMemorysChange,val:this});//UI再描画用
    };
    _AbstractDevice_inputctl_MTControlKit.prototype._SetPlayPosition=function(num){
        var obj=this._PossionMemorys[UTL_INT(num)];
        if(obj instanceof Object) {
            var key=Object.keys(obj);
            for(var i=0;i<key.length;i++) {
                this.Binder_Motor.Par_Val = new ST_CtlProtocol("SetPosition",new ST_MTRotState(obj[key[i]], UTL_conv_rpm_to_radi_sec(this._PosSpeed), this._PosTorque) ,key[i]);//x_speed:RPM ->  (radians/second) に
            }
        }
    };

    //Speed制御の有効化
    _AbstractDevice_inputctl_MTControlKit.prototype._SetSpeedModeActive=function(flg){
        flg= flg==undefined?!this._IsSpeedModePlay:flg;//引数無しでトグル
        this._IsSpeedModePlay=flg;
        this._IsPositionModePlay=false;
        if(!flg){
            this._SendStop();
        }else{
            this._SendSpeedMode();
        }
        this.EventHandler.fire({type:this.EventHandlerType.OnModePlayChange,val:this});//UI再描画用
    };
    //Position制御の有効化
    _AbstractDevice_inputctl_MTControlKit.prototype._SetPositionModeActive=function(flg){
        flg= flg==undefined?!this._IsPositionModePlay:flg;//引数無しでトグル
        this._IsPositionModePlay=flg;
        this._IsSpeedModePlay=false;
        if(!flg){
            this._SendStop();
        }else{
            this._SendPositionMode();
        }
        this.EventHandler.fire({type:this.EventHandlerType.OnModePlayChange,val:this});//UI再描画用
    };

    _AbstractDevice_inputctl_MTControlKit.prototype._SetSpeed=function(val){
        this._Speed=val;
        this._SendSpeedMode();
    };
    _AbstractDevice_inputctl_MTControlKit.prototype._SetPosSpeed=function(val){
        this._PosSpeed=val;
        this._SendPositionMode();
    };
    _AbstractDevice_inputctl_MTControlKit.prototype._SetPosTorque=function(val){
        this._PosTorque=val;
        this._SendPositionMode();
    };
    _AbstractDevice_inputctl_MTControlKit.prototype._SetPosPosition=function(val){
        this._PosPosition=val;
        this._SendPositionMode();
    };


    _AbstractDevice_inputctl_MTControlKit.prototype._SendSpeedMode=function(){
        if(this._IsSpeedModePlay) {
            this.Binder_Motor.Par_Val = new ST_CtlProtocol("SetSpeed_rpm", this._Speed);
        }
    };
    _AbstractDevice_inputctl_MTControlKit.prototype._SendPositionMode=function(){
        if(this._IsPositionModePlay){
            this.Binder_Motor.Par_Val=new ST_CtlProtocol("SetPosition",
                [this._PosPosition, UTL_conv_rpm_to_radi_sec(this._PosSpeed),this._PosTorque]);
        }
    };
    _AbstractDevice_inputctl_MTControlKit.prototype._SendStop=function(){
        this.Binder_Motor.Par_Val=new ST_CtlProtocol("HardStop",true);
    };

    _AbstractDevice_inputctl_MTControlKit.prototype._SendPreSetPositon=function(){
        //todo::現状モーター側でbleMotorPreSetPositonを実行しても、モーターから通知されるnotifyのPostionが0にならない
        this.Binder_Motor.Par_Val=new ST_CtlProtocol("PreSetPositon",0);
    };

//----------------------------------------------------------------------------------------------------------------//
//  section::##### モジュール抽象クラス
//----------------------------------------------------------------------------------------------------------------//
    /************************************************************************************
     * section::***** モジュール（基底クラス）
     * ●プロパティ
     * 　
     * ●イベント
     ************************************************************************************/
    function _AbstractDevice_module_base(option,class_option){
        "use strict";
        var self=this;
        _AbstractDevice_base.apply(this, arguments);

        ///破棄時のイベント（基底クラスから呼び出される）　
        this.EventHandler.addListener(this.EventHandlerType.OnDestory,function(e){
        });

        self.status=true;//info::モジュールは全てtrue固定
    }
    //デバイス基底クラス継承
    _AbstractDevice_module_base.prototype = Object.create(_AbstractDevice_base.prototype,{constructor:{value:_AbstractDevice_module_base}});


    /************************************************************************************
     *  section::***** val_branch　二股分岐　RootのデータをBranch_A/Bに分岐 or 統合する。
     *  info::Rootを最初にバインドしないと型が決まらない為、Branch_A、Branch_Bはバインド出来ない
         ●バインダー　Binder_Rootを最初にバインドしないと型が決まらない為、Out_A、Out_Bはバインド出来ない
             ◎Root	(表示名:Root)	//・分岐元 or 統合先  Branch_A/Bより前にここに最初に接続する
             ◎Branch_A	(表示名:Branch_A)	//・分岐A
             ◎Branch_B	(表示名:Branch_B)	//・分岐B
         ●プロパティ 無し
     ************************************************************************************/
    function _AbstractDevice_val_branch(option,class_option) {
        "use strict";
        _AbstractDevice_module_base.apply(this, arguments);//info::基底クラスが不適切
        var self = this;
        //--------------------//
        //section::追加イベント
        //--------------------//
        //--------------------//
        //section::内部プロパティ
        //--------------------//
        //--------------------//
        //section::保存プロパティの復帰＆デフォルト値設定
        //--------------------//
        //--------------------//
        //section::バインダーの生成
        //--------------------//

        //入力値
        self.Binder_Root=this.CreateBinder({
            binder_name:"Root"
            ,binder_show_name:"Root" //Binder_Branch_ABより前にここに最初に接続する
            ,type_list: DeviceValBinder_TYPE_LIST.VAL_ALL
            ,allow_list:DeviceValBinder_TYPE_LIST.VAL_ALL
            ,dataflowcb_in:function(val){
                //Binder_Branch_A、Branch_Bに出力
                self.Binder_Branch_A.Par_Val=val;
                self.Binder_Branch_B.Par_Val = val;
            }
            ,dataflowcb_get:null
            ,is_dataflow_out:true
            ,is_dataflow_pull:false
            ,option:option
        });
        //バインド切り替え時
        self.Binder_Root.EventHandler.addListener(self.Binder_Root.EventHandlerType.OnBindChange,function(e){
            _bindcheck();
        });

        //出力先A
        self.Binder_Branch_A=this.CreateBinder(
            {
                binder_name:"Branch_A"
                ,binder_show_name:"Branch A"
                ,type_list:0
                ,allow_list:0
                ,dataflowcb_in:function(val){
                    //Binder_Rootに出力
                    self.Binder_Root.Par_Val=val;
                }
                ,dataflowcb_get:null
                ,is_dataflow_out:true
                ,is_dataflow_pull:false
                ,option:option
            }
        );
        //バインド切り替え時
        self.Binder_Branch_A.EventHandler.addListener(self.Binder_Branch_A.EventHandlerType.OnBindChange,function(e){
            _bindcheck();
        });

        //出力先B
        self.Binder_Branch_B=this.CreateBinder(
            {
                binder_name:"Branch_B"
                ,binder_show_name:"Branch B"
                ,type_list: 0
                ,allow_list:0
                ,dataflowcb_in:function(val){
                    //Binder_Rootに出力
                    self.Binder_Root.Par_Val=val;
                }
                ,dataflowcb_get:null
                ,is_dataflow_out:true
                ,is_dataflow_pull:false
                ,option:option
            }
        );
        //バインド切り替え時
        self.Binder_Branch_B.EventHandler.addListener(self.Binder_Branch_B.EventHandlerType.OnBindChange,function(e){
            _bindcheck();
        });

        function _bindcheck() {
            if(self.Binder_Root.is_bind && !self.Binder_Branch_A.is_bind && !self.Binder_Branch_B.is_bind) {
                //info::バインダの型変更 1) ABがnullでRootをバインドした場合はABをROOTと同じ型に
                self.Binder_Branch_A._allow_Comparator = self.Binder_Branch_B._allow_Comparator = self.Binder_Root._binding_binder_ref._allow_Comparator;
                self.Binder_Branch_A._type_list = self.Binder_Branch_B._type_list = self.Binder_Root._binding_binder_ref._type_list;
                self.Binder_Branch_A._dataflow_list = self.Binder_Branch_B._dataflow_list = self.Binder_Root._binding_binder_ref._dataflow_list;
            }else if(self.Binder_Root.is_bind && (self.Binder_Branch_A.is_bind || self.Binder_Branch_B.is_bind)){
                //info::バインダの型変更 2) Rootをバインド後ABをどちらかを繋いだ場合、Rootの型をABのどちらか接続済みの型と同じにする(両方接続済みの場合はAを優先)
                var source_binding_binder_ref=self.Binder_Branch_A.is_bind?self.Binder_Branch_A._binding_binder_ref:self.Binder_Branch_B._binding_binder_ref;
                self.Binder_Root._type_list = source_binding_binder_ref._type_list;
                self.Binder_Root._allow_Comparator = source_binding_binder_ref._allow_Comparator;
                self.Binder_Root._dataflow_list =source_binding_binder_ref._dataflow_list;

            }else if(!self.Binder_Root.is_bind && !self.Binder_Branch_A.is_bind&& !self.Binder_Branch_B.is_bind){
                //info::バインダの型変更 3) 全バインダーが空の場合、デフォルト(Root:All AB:型無し接続不可)に戻す
                self.Binder_Root._type_list=self.Binder_Branch_A._type_list=self.Binder_Branch_B._type_list=DeviceValBinder_TYPE_LIST.VAL_ALL;
                self.Binder_Root._allow_Comparator=self.Binder_Branch_A._allow_Comparator=self.Binder_Branch_B._allow_Comparator=DeviceValBinder_TYPE_LIST.VAL_ALL;
                self.Binder_Root._dataflow_list=self.Binder_Branch_A._allow_Comparator=self.Binder_Branch_B._allow_Comparator=(DeviceValBinder_DATA_FLOW_LIST.DATA_FLOW_IN||DeviceValBinder_DATA_FLOW_LIST.DATA_FLOW_OUT);
            }
        }
        //--------------------//
        //  初期化
        //--------------------//
        // -------------------//
        //  パブリックプロパティ
        //--------------------//
        //--------------------//
        //  編集用UIに公開するプロパティ
        //--------------------//
        // -------------------//
        //  イベントバインド
        //--------------------//
        ///破棄時のイベント（基底クラスから呼び出される）　
        this.EventHandler.addListener(self.EventHandlerType.OnDestory,function(e){});
        ////

    }
    //基底クラス継承
    _AbstractDevice_val_branch.prototype = Object.create(_AbstractDevice_module_base.prototype,{constructor:{value:_AbstractDevice_val_branch}});

    /************************************************************************************
     * section::*****  SingleWheelCar_Assembly 1輪用(1モーター)コントローラー
     *
         ●バインダー
             ◎Control_Vector	(表示名:ControlVector)	//・進行方向のベクトル入力
             ◎Control_Deg	(表示名:Control Degree)	//・進行方向の角度入力
             ◎Motor_X	(表示名:Motor)	//・モーターの制御プロトコル(モーターに接続)
             ◎RecCtl	(表示名:RecCtl Input)	//・位置を記憶するコントローラーの入力
             ◎Brake	(表示名:Brake)	//・true:急停止-->ロック false:急停止-->ニュートラル
             ◎RecPosition『Advance』	(表示名:RecPosition)	//・記憶位置を保存するメモリ番号
             ◎PlayPosition『Advance』	(表示名:PlayPosition)	//・記憶位置を再生するメモリ番号
             ◎PossionMemorys『Advance』	(表示名:PossionMemorys)	//・記憶した位置座標の値  null "" undefindeを指定した場合は全消去
             ◎ResetIntegPosition『Advance』	(表示名:Reset IntegPosition)	//・モーターの相対位置をリセットする
             ◎ChangePossionMemorysNotify『Advance』	(表示名:Change PossionMemorys Notify)	//・記憶した位置座標の値 RecPositionで変更の度に送信される
             ◎ClearPossionMemorys『Advance』	(表示名: Clear PossionMemorys)	//・記憶した位置座標の全消去
         ●プロパティ
             ◎ MaxSpeed 『Advance』		//・最大速度値rpm
             ◎ RecMoveSpeed		//・記憶位置再生時の移動速度 0-300rpm
             ◎ ReversMotor		//・回転方向の反転
             ◎ SetRecPosition『Advance』		//・記憶位置を保存するメモリ番号
             ◎ SetPlayPosition『Advance』		//・記憶位置を再生するメモリ番号
             ◎ SetResetPosition『Advance』		//・現在のモーター位置を0として積算位置をリセット
     ************************************************************************************/
    function _AbstractDevice_SingleWheelCar_Assembly(option,class_option) {
        "use strict";
        _AbstractDevice_module_base.apply(this, arguments);
        var self = this;

        //--------------------//
        //section::追加イベントハンドラ定義
        //--------------------//
        //--------------------//
        //section::内部プロパティ
        //--------------------//
        this._Rev_MTRotState_X={};//モーターの状態X notifyに連動
        this._LedStateColor_X=new ST_MTLedState();//LEDの色
        //--------------------//
        //section::保存プロパティの復帰＆デフォルト値設定
        //--------------------//

        this.PROP.MaxSpeed=option.MaxSpeed?option.MaxSpeed:100;
        this.PROP.RecMoveSpeed=option.RecMoveSpeed?option.RecMoveSpeed:50;
        this.PROP.ReversMotor=option.ReversMotor?true:false;
        this.PROP.PossionMemorys=option.PossionMemorys?option.PossionMemorys:[];//保存位置
        //前回のモーターのリセット位置
       // this.PROP.PreSetPositon_Motor_x= option.PreSetPositon_Motor_x instanceof Object ?option.PreSetPositon_Motor_x:{};
        //--------------------//
        //section::バインダーの生成
        //--------------------//
        self.Binder_Control_Vector=this.CreateBinder(
            {
                binder_name:"Control_Vector"
                ,binder_show_name:'Direction Vector'
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT|DeviceValBinder_TYPE_LIST.VAL_VECTOR2
                ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_ROBOT^DeviceValBinder_TYPE_LIST.DEVICE_ALL)|DeviceValBinder_TYPE_LIST.VAL_VECTOR2
                ,dataflowcb_in:function(vector2){
                    self.SetInput=vector2;
                }
                ,dataflowcb_get:null
                ,is_dataflow_out:false
                ,is_dataflow_pull:false
                ,option:option
            }
        );
        self.Binder_Control_Rad=this.CreateBinder(
            {
                binder_name:"Control_Rad"
                ,binder_show_name:'Direction Radian'
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT|DeviceValBinder_TYPE_LIST.VAL_NUMBER
                ,allow_list:DeviceValBinder_TYPE_LIST.VAL_NUMBER
                ,dataflowcb_in:function(number){
                self.SetRadian=number;
            }
                ,dataflowcb_get:null
                ,is_dataflow_out:false
                ,is_dataflow_pull:false
                ,option:option
            }
        );
        self.Binder_Control_Deg=this.CreateBinder(
            {
                binder_name:"Control_Deg"
                ,binder_show_name:'Direction Degree'
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT|DeviceValBinder_TYPE_LIST.VAL_NUMBER
                ,allow_list:DeviceValBinder_TYPE_LIST.VAL_NUMBER
                ,dataflowcb_in:function(number){
                    self.SetDegree=number;
                }
                ,dataflowcb_get:null
                ,is_dataflow_out:false
                ,is_dataflow_pull:false
                ,option:option
            }
        );
        self.Binder_Motor_X=this.CreateBinder(
            {
                binder_name:"Motor"
                ,binder_show_name:"Motor: Axis X"
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT | DeviceValBinder_TYPE_LIST.VAL_CTL_MT
                ,allow_list:DeviceValBinder_TYPE_LIST.DEVICE_MOTOR | DeviceValBinder_TYPE_LIST.VAL_CTL_MT
                ,dataflowcb_in:function(val){
                    if(val instanceof ST_NotifyProtocol){
                        if(val.Val instanceof ST_MTRotState){
                            //回転情報受信
                            self._Rev_MTRotState_X[val.originID]=val.Val;
                        }else if(val.Val instanceof ST_MTLedState){
                            //LED情報受信
                            self._LedStateColor_X=val.Val;
                            //AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnIconViewLedStateReDrowCall,device:self});//LEDアイコン再描画用
                            AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnIconViewReDrowCall,device:self});//アイコン再描画用
                        }
                    }
            }
                ,dataflowcb_get:null
                ,is_dataflow_out:true
                ,is_dataflow_pull:false
                ,option:option
            }
        );


        self.Binder_RecCtl=this.CreateBinder(
            {
                binder_name:"RecCtl"
                ,binder_show_name:"Position Recorder Input"
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT|DeviceValBinder_TYPE_LIST.VAL_CTL_REC
                ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_ROBOT^DeviceValBinder_TYPE_LIST.DEVICE_ALL)|DeviceValBinder_TYPE_LIST.VAL_CTL_REC
                ,dataflowcb_in:function(val){
                //Commandに対応するプロパティを実行 (info:: UTL_IsSetterOnePerentProto：setterプロパティの動的指定時に、setterと間違えて関数や定数の参照を上書きするのを防止する為に使用)
                if(val instanceof ST_CtlProtocol && UTL_IsSetterOnePerentProto(self,val.Command)){
                    //command指定時にこの値があると、指定したIDの仮想デバイスのみ実行
                    if(val.targetID&&val.targetID!==self.UID){
                        return;
                    }
                    self[val.Command]=val.Val;
                }
            }
                ,dataflowcb_get:function(){
                return new ST_NotifyProtocol("PossionMemorys", self.PossionMemorys, self.UID);
            }
                ,is_dataflow_out:true
                ,is_dataflow_pull:false
                ,option:option
            }
        );

        self.Binder_brake=this.CreateBinder({
            binder_name:"Brake"
            ,binder_show_name:"Brake"
            ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT|DeviceValBinder_TYPE_LIST.VAL_BOOL
            ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_ROBOT^DeviceValBinder_TYPE_LIST.DEVICE_ALL)|DeviceValBinder_TYPE_LIST.VAL_NUMBER
            ,dataflowcb_in:function(val){
                self.SetBrake=val;
            }
            ,dataflowcb_get:null
            ,is_dataflow_out:false
            ,is_dataflow_pull:false
            ,option:option
        });

        self.Binder_RecPosition=this.CreateBinder(
            {
                binder_name:"RecPosition"
                ,binder_show_name:"Record Position at Address"
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT|DeviceValBinder_TYPE_LIST.VAL_INT
                ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_ROBOT^DeviceValBinder_TYPE_LIST.DEVICE_ALL)|DeviceValBinder_TYPE_LIST.VAL_INT | DeviceValBinder_TYPE_LIST.VAL_FLOAT
                ,dataflowcb_in:function(val){
                self.SetRecPosition=val;
            }
                ,dataflowcb_get:null
                ,is_dataflow_out:false
                ,is_dataflow_pull:false
                ,ui_show_type:DeviceValBinder_UI_SHOW_TYPE.Advance
                ,option:option
            }
        );
        self.Binder_PlayPosition=this.CreateBinder(
            {
                binder_name:"PlayPosition"
                ,binder_show_name:"Play Position at Address"
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT|DeviceValBinder_TYPE_LIST.VAL_INT
                ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_ROBOT^DeviceValBinder_TYPE_LIST.DEVICE_ALL)|DeviceValBinder_TYPE_LIST.VAL_INT | DeviceValBinder_TYPE_LIST.VAL_FLOAT
                ,dataflowcb_in:function(val){
                self.SetPlayPosition=val;
            }
                ,dataflowcb_get:null
                ,is_dataflow_out:false
                ,is_dataflow_pull:false
                ,ui_show_type:DeviceValBinder_UI_SHOW_TYPE.Advance
                ,option:option
            }
        );
        self.Binder_PossionMemorys=this.CreateBinder(
            {
                binder_name:"PossionMemorys"
                ,binder_show_name:"Positions in Memory"
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT|DeviceValBinder_TYPE_LIST.VAL_ARRAY
                ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_ROBOT^DeviceValBinder_TYPE_LIST.DEVICE_ALL)|DeviceValBinder_TYPE_LIST.VAL_ARRAY | DeviceValBinder_TYPE_LIST.VAL_FLOAT | DeviceValBinder_TYPE_LIST.VAL_INT
                ,dataflowcb_in:function(val){self.PossionMemorys=val;}
                ,dataflowcb_get:function(){ return self.PossionMemorys;}
                ,is_dataflow_out:false
                ,is_dataflow_pull:false
                ,ui_show_type:DeviceValBinder_UI_SHOW_TYPE.Advance
                ,option:option
            }
        );
        //info::単なるトリガー 値の設定取得は無効
        //モーターの相対位置をリセットする
        self.Binder_ResetIntegPosition=this.CreateBinder(
            {
                binder_name:"ResetIntegPosition"
                ,binder_show_name:"Reset Position"
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT|DeviceValBinder_TYPE_LIST.VAL_BOOL
                ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_ROBOT^DeviceValBinder_TYPE_LIST.DEVICE_ALL)|DeviceValBinder_TYPE_LIST.VAL_NUMBER
                ,dataflowcb_in:function(val){
                self.SetResetPosition=val;
            }
                ,dataflowcb_get:null
                ,is_dataflow_out:false
                ,is_dataflow_pull:false
                ,ui_show_type:DeviceValBinder_UI_SHOW_TYPE.Advance
                ,option:option
            }
        );
        self.Binder_Change_PossionMemorys=this.CreateBinder(
            {
                binder_name:"ChangePossionMemorysNotify"
                ,binder_show_name:"Notify Change of Position Memory"
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT|DeviceValBinder_TYPE_LIST.VAL_ARRAY
                ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_ROBOT^DeviceValBinder_TYPE_LIST.DEVICE_ALL)|DeviceValBinder_TYPE_LIST.VAL_ARRAY
                ,dataflowcb_in:null
                ,dataflowcb_get:function(){
                    return self.PossionMemorys;//info::Gettrの他にはBinder_RecPositionで変更の度に送信される
                }
                ,is_dataflow_out:true
                ,is_dataflow_pull:false
                ,ui_show_type:DeviceValBinder_UI_SHOW_TYPE.Advance
                ,option:option
            }
        );
        self.Binder_ClearPossionMemorys=this.CreateBinder(
            {
                binder_name:"ClearPossionMemorys"
                ,binder_show_name:"Clear All Position Memory"
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT|DeviceValBinder_TYPE_LIST.VAL_BOOL
                ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_ROBOT^DeviceValBinder_TYPE_LIST.DEVICE_ALL)|DeviceValBinder_TYPE_LIST.VAL_NUMBER
                ,dataflowcb_in:function(val){
                self.PossionMemorys=null;
                //self.SetClearPosition=val;
            }
                ,dataflowcb_get:null
                ,is_dataflow_out:false
                ,is_dataflow_pull:false
                ,ui_show_type:DeviceValBinder_UI_SHOW_TYPE.Advance
                ,option:option
            }
        );


        //--------------------//
        //section::初期化
        //--------------------//

        //--------------------//
        //section::編集用UIに公開するプロパティ
        //--------------------//

        this.PublishUIProp({prop_name:"MaxSpeed",prop_show_name:"Max Speed",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:0,maxlim:300,step:1});//rpm
        this.PublishUIProp({prop_name:"RecMoveSpeed",prop_show_name:"Playback Speed",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:0,maxlim:300,step:1});//rpm
        this.PublishUIProp({prop_name:"ReversMotor",prop_show_name:"Reverse",prop_type:PublishUIProp_TYPE.BOOL,lowlim:null,maxlim:null,step:null});
        this.PublishUIProp({prop_name:"SetRecPosition",prop_show_name:"Record Position at Address",prop_type:PublishUIProp_TYPE.INT,lowlim:null,maxlim:null,step:null,ui_show_type:PublishUIProp_UI_SHOW_TYPE.Advance});
        this.PublishUIProp({prop_name:"SetPlayPosition",prop_show_name:"Play Position at Address",prop_type:PublishUIProp_TYPE.INT,lowlim:null,maxlim:null,step:null,ui_show_type:PublishUIProp_UI_SHOW_TYPE.Advance});
        this.PublishUIProp({prop_name:"SetResetPosition",prop_show_name:"Reset Position",prop_type:PublishUIProp_TYPE.SEND_NULL,lowlim:null,maxlim:null,step:null,ui_show_type:PublishUIProp_UI_SHOW_TYPE.Advance});
        //this.PublishUIProp({prop_name:"SetClearPosition",prop_show_name:"SetClearPosition",prop_type:PublishUIProp_TYPE.INT,lowlim:0,maxlim:10,step:1});
        // -------------------//
        //section::イベントバインド
        //--------------------//
        //バインド切り替え時
        self.Binder_Motor_X.EventHandler.addListener(self.Binder_Motor_X.EventHandlerType.OnBindChange,function(e){
            //LED情報の取得要求
            self.Binder_Motor_X.Par_Val=new ST_CtlProtocol("GetLedStateColorNotifyOne",true);
        });
        ///破棄時のイベント（基底クラスから呼び出される）　
        this.EventHandler.addListener(self.EventHandlerType.OnDestory,function(e){
        });
    }
    //基底クラス継承
    _AbstractDevice_SingleWheelCar_Assembly.prototype = Object.create(_AbstractDevice_module_base.prototype,{constructor:{value:_AbstractDevice_SingleWheelCar_Assembly}});
    // -------------------//
    // section::定数定義
    //--------------------//
    // -------------------//
    // section::パブリックプロパティ
    //--------------------//

    Object.defineProperty(_AbstractDevice_SingleWheelCar_Assembly.prototype,'MaxSpeed', {
        get: function() {
            return this.PROP.MaxSpeed;
        },
        set:function(val){
            this.PROP.MaxSpeed=val;
        }
    });
    Object.defineProperty(_AbstractDevice_SingleWheelCar_Assembly.prototype,'RecMoveSpeed', {
        get: function() {
            return this.PROP.RecMoveSpeed;
        },
        set:function(val){
            this.PROP.RecMoveSpeed=val;
        }
    });
    //
    //出力値反転
    Object.defineProperty(_AbstractDevice_SingleWheelCar_Assembly.prototype,'ReversMotor', {
        get: function() {
            return this.PROP.ReversMotor;
        },
        set:function(val){
            this.PROP.ReversMotor=val;
        }
    });
    //入力されたベクトル値をモーターに送る vector2
    Object.defineProperty(_AbstractDevice_SingleWheelCar_Assembly.prototype,'SetInput', {
        get: function() {
            return null;
        },
        set:function(vector2){
            if(! (vector2 instanceof ST_Vector2)){return;}
            //ベクトル値による分配
            var _rpm=vector2.x*this.PROP.MaxSpeed;
            _rpm= Number.isFinite(_rpm)?_rpm:0;

            //モーターへ送信
            this.Binder_Motor_X.Par_Val=new ST_CtlProtocol("SetSpeed_rpm",_rpm);
        }
    });
    //入力された角度値(rad)をモーターに送る float
    Object.defineProperty(_AbstractDevice_SingleWheelCar_Assembly.prototype,'SetRadian', {
        get: function() {
            return null;
        },
        set:function(number){
            var _rad=UTL_N(number);
            var rotst=new ST_MTRotState(_rad, this.MaxSpeed,ST_MTRotState.MAX_TORQUE);//x_speed:RPM ->  (radians/second) に
            //モーターへ送信
            this.Binder_Motor_X.Par_Val = new ST_CtlProtocol("SetPosition",rotst);
        }
    });
    //入力された角度値(deg)をモーターに送る float
    Object.defineProperty(_AbstractDevice_SingleWheelCar_Assembly.prototype,'SetDegree', {
        get: function() {
            return null;
        },
        set:function(number){
            var _deg=UTL_N(number);
            var rotst=new ST_MTRotState(_deg, this.MaxSpeed,ST_MTRotState.MAX_TORQUE);//x_speed:RPM ->  (radians/second) に
            //モーターへ送信
            this.Binder_Motor_X.Par_Val = new ST_CtlProtocol("SetPosition_deg",rotst);
        }
    });

    Object.defineProperty(_AbstractDevice_SingleWheelCar_Assembly.prototype,'SetBrake', {
        get: function() {return null;},
        set:function(val){
            var cmd= !val?"HardStop":"HardBrake"; //true:急停止>ニュートラルモード false:急停止＆ロックモード　
            this.Binder_Motor_X.Par_Val=new ST_CtlProtocol(cmd,true);
        }
    });

    //現在の位置を0として積算位置をリセット
    Object.defineProperty(_AbstractDevice_SingleWheelCar_Assembly.prototype,'SetResetPosition', {
        get: function() {
            return null;
        },
        set:function(val){
                this._inVector=new ST_Vector2();
                //モーターの絶対位置をリセット
                // var key=Object.keys(this._Rev_MTRotState_X);
                // for(var i=0;i<key.length;i++) {
                //     this.PROP.PreSetPositon_Motor_x[key[i]]=0;
                // }
                this.Binder_Motor_X.Par_Val=new ST_CtlProtocol("PreSetPositon",0);
        }
    });
    //SetRecPosition(W):int[0 - n] 記憶位置を保存するメモリ番号
    Object.defineProperty(_AbstractDevice_SingleWheelCar_Assembly.prototype,'SetRecPosition', {
        get: function() {return null;
        },
        set:function(val){
            var x_pos={};
            var key=Object.keys(this._Rev_MTRotState_X);
            for(var i=0;i<key.length;i++) {
                var k=key[i];
               // x_pos[k]=UTL_N(this._Rev_MTRotState_X[k].Position-this.PROP.PreSetPositon_Motor_x[k]);
                x_pos[k]=UTL_N(this._Rev_MTRotState_X[k].Position);
            }

            this.PROP.PossionMemorys[UTL_INT(val)]={x:x_pos};
            this.Binder_RecCtl.Par_Val=new ST_NotifyProtocol("PossionMemorys",this.PossionMemorys,this.UID);//記憶情報の送信
            this.Binder_Change_PossionMemorys.Par_Val=this.PossionMemorys;
        }

    });
    //SetPlayPosition(W):int[0 - 4]記憶位置を再生するメモリ番号
    Object.defineProperty(_AbstractDevice_SingleWheelCar_Assembly.prototype,'SetPlayPosition', {
        get: function() {
            return null;
        },
        set:function(val){
            var obj=this.PROP.PossionMemorys[UTL_INT(val)];
            if(obj){
                this._SetMorterPosition(obj.x);
            }
        }
    });

    //SetClearPosition(W):int[0 - 4]記憶位置を消去するメモリ番号 undefinde nullを指定した場合は全消去
    Object.defineProperty(_AbstractDevice_SingleWheelCar_Assembly.prototype,'SetClearPosition', {
        get: function() {
            return null;
        },
        set:function(val){
            if(val==null){
                this.PossionMemorys=null;
            }else{
                this.PROP.PossionMemorys[UTL_INT(val)]=null;
                this.Binder_RecCtl.Par_Val=new ST_NotifyProtocol("PossionMemorys",this.PossionMemorys,this.UID);//記憶情報の送信
                this.Binder_Change_PossionMemorys.Par_Val=this.PossionMemorys;
            }
        }
    });

    //PossionMemorys(RW)
    Object.defineProperty(_AbstractDevice_SingleWheelCar_Assembly.prototype,'PossionMemorys', {
        get: function() {
            //PossionMemorysの複製を返す
            var mem=this.PROP.PossionMemorys;
            var rt=[];
            for(var i=0;i<mem.length;i++){
                if(mem[i] instanceof Object){
                    rt[i]=UTL_clone(mem[i]);
                }
            }
            return rt;
        },
        set:function(val){
            this.PROP.PossionMemorys=[];
            if(val instanceof Array){
                for(var i=0;i=val.length;i++){
                    this.PROP.PossionMemorys[i]=val[i];
                }
            }
            this.Binder_RecCtl.Par_Val=new ST_NotifyProtocol("PossionMemorys",this.PossionMemorys,this.UID);//記憶情報の送信
            this.Binder_Change_PossionMemorys.Par_Val=this.PossionMemorys;
        }
    });


    // -------------------//
    // section::内部関数
    //--------------------//

    //radianでモーターの位置を設定する（メモリ設定用）
    _AbstractDevice_SingleWheelCar_Assembly.prototype._SetMorterPosition=function(x_radians){
        if(x_radians instanceof Object) {
            var key=Object.keys(x_radians);
            var speed=this.RecMoveSpeed;
            for(var i=0;i<key.length;i++) {
                this.Binder_Motor_X.Par_Val = new ST_CtlProtocol("SetPosition", [x_radians[key[i]], UTL_conv_rpm_to_radi_sec(speed), 0],key[i]);//x_speed:RPM ->  (radians/second) に
            }
        }
    };


    /************************************************************************************
     *  section::***** TwoWheelCar_Assembly 2輪用(2モーター)コントローラー
     *  ベクトル入力値から前進後退回転・停止の制御を行う
         ●バインダー
             ◎Control_Vector	(表示名:ControlVector)	//・進行方向のベクトル入力
             ◎Motor_L	(表示名:Motor_L)	//・モーターの制御プロトコル(左側モーターに接続)
             ◎Motor_R	(表示名: Motor_R)	//・モーターの制御プロトコル(右側モーターに接続)
             ◎Brake	(表示名:Brake)	//・true:急停止-->ニュートラル false:急停止-->ロック
         ●プロパティ
             ◎ MaxSpeed		//・最大速度 10-300rpm
             ◎ DistRatio		//・速度の変化比率 0-2
             ◎ ReversMotor_L		//・回転方向の反転(左側)
             ◎ ReversMotor_R		//・回転方向の反転(右側)
     ************************************************************************************/
    function _AbstractDevice_TwoWheelCar_Assembly(option,class_option) {
        "use strict";
        _AbstractDevice_module_base.apply(this, arguments);
        var self = this;

        //--------------------//
        //section::追加イベントハンドラ定義
        //--------------------//
        //--------------------//
        //section::内部プロパティ
        //--------------------//
        this._Lval=0;
        this._Rval=0;
        this._inVector=new ST_Vector2();
        this._Rev_MTRotState_L=new ST_NotifyProtocol();//モーターの回転情報L notifyに連動
        this._Rev_MTRotState_R=new ST_NotifyProtocol();//モーターの回転情報R
        this._LedStateColor_L=new ST_MTLedState();//LEDの色 モジュールアイコンSVGの色で使用
        this._LedStateColor_R=new ST_MTLedState();//LEDの色 モジュールアイコンSVGの色で使用
        //--------------------//
        //section::保存プロパティの復帰＆デフォルト値設定
        //--------------------//
        this.PROP.MaxSpeed=option.MaxSpeed?option.MaxSpeed:100;
        this.PROP.ReversMotor_L=option.ReversMotor_L===undefined?false:(option.ReversMotor_L?true:false);
        this.PROP.ReversMotor_R=option.ReversMotor_R===undefined?true:(option.ReversMotor_R?true:false);
        this.PROP.DistRatio=option.DistRatio?option.DistRatio:1;
        //--------------------//
        //section::バインダーの生成
        //--------------------//
        self.Binder_Control_Vector=this.CreateBinder(
            {
                binder_name:"Control_Vector"
                ,binder_show_name:'Direction Vector'
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT|DeviceValBinder_TYPE_LIST.VAL_VECTOR2
                ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_ROBOT^DeviceValBinder_TYPE_LIST.DEVICE_ALL)|DeviceValBinder_TYPE_LIST.VAL_VECTOR2
                ,dataflowcb_in:function(vector2){
                    self._inVector=vector2.Clone();
                    var sθ=self._inVector.RadianFromZero();
                    var sx=self._inVector.x;
                    var sr=self._inVector.DistanceFromZero();

                    //θ角から4現象の区分け info::もっと効率の良いロジックがあるハズ
                    var ans_x=0,ans_y=0;

                    //info::速度付き&十字キー改訂ロジック
                    switch(true){
                        case sθ <= -Math.PI*0.5://第3現象 (-2x-r,-r)
                            ans_x= 2*sx*sx/(sr*self.DistRatio) -sr;ans_y=-sr ;
                            break;
                        case sθ <= 0://第4現象 (-r,2x-r)
                            ans_x= -sr;ans_y=2*sx*sx/(sr*self.DistRatio) -sr ;
                            break;
                        case sθ <= Math.PI*0.5://第1現象 (r,-2x^2/r+r)
                            ans_x= sr; ans_y= -2*sx*sx/(sr*self.DistRatio) +sr;
                            break;
                        case sθ <= Math.PI://第2現象 (-2x^2/r+r,r)
                            // ans_x= 2*sx+sr;ans_y= sr;
                            ans_x= -2*sx*sx/(sr*self.DistRatio) +sr;ans_y= sr;
                            break;
                    }
                    ans_x=Number.isFinite(ans_x)?ans_x:0;
                    ans_y=Number.isFinite(ans_y)?ans_y:0;

                    self._Lval=ans_x*self.PROP.MaxSpeed;
                    self._Rval=ans_y*self.PROP.MaxSpeed;


                    var _L_rpm=self.ReversMotor_L?self._Lval*-1:self._Lval;
                    var _R_rpm=self.ReversMotor_R?self._Rval*-1:self._Rval;
                    //モーターへ送信
                    self.Binder_Motor_L.Par_Val=new ST_CtlProtocol("SetSpeed_rpm",_L_rpm);
                    self.Binder_Motor_R.Par_Val=new ST_CtlProtocol("SetSpeed_rpm",_R_rpm);
                }
                ,dataflowcb_get:null
                ,is_dataflow_out:false
                ,is_dataflow_pull:false
                ,option:option
            }
        );
        self.Binder_Motor_L=this.CreateBinder(
            {
                binder_name:"Motor_L"
                ,binder_show_name:"Motor: Wheel Left"
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT | DeviceValBinder_TYPE_LIST.VAL_CTL_MT
                ,allow_list:DeviceValBinder_TYPE_LIST.DEVICE_MOTOR | DeviceValBinder_TYPE_LIST.VAL_CTL_MT
                ,dataflowcb_in:function(val){
                    if(val instanceof ST_NotifyProtocol){
                        if(val.Val instanceof ST_MTRotState){
                            //回転情報受信
                            self._Rev_MTRotState_L[val.originID]=val.Val;
                        }else if(val.Val instanceof ST_MTLedState){
                            //LED情報受信
                            self._LedStateColor_L=val.Val;
                            //AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnIconViewLedStateReDrowCall,device:self});//LEDアイコン再描画用
                            AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnIconViewReDrowCall,device:self});//アイコン再描画用
                        }
                    }
                }
                ,dataflowcb_get:null
                ,is_dataflow_out:true
                ,is_dataflow_pull:false
                ,option:option
            }
        );
        self.Binder_Motor_R=this.CreateBinder({
                binder_name:"Motor_R"
                ,binder_show_name:"Motor: Wheel Right"
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT | DeviceValBinder_TYPE_LIST.VAL_CTL_MT
                ,allow_list:DeviceValBinder_TYPE_LIST.DEVICE_MOTOR | DeviceValBinder_TYPE_LIST.VAL_CTL_MT
                ,dataflowcb_in:function(val){
                    if(val instanceof ST_NotifyProtocol){
                        if(val.Val instanceof ST_MTRotState){
                            //回転情報受信
                            self._Rev_MTRotState_R[val.originID]=val.Val;
                        }else if(val.Val instanceof ST_MTLedState){
                            //LED情報受信
                            self._LedStateColor_R=val.Val;
                            //AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnIconViewLedStateReDrowCall,device:self});//LEDアイコン再描画用
                            AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnIconViewReDrowCall,device:self});//アイコン再描画用
                        }
                    }
                }
                ,dataflowcb_get:null
                ,is_dataflow_out:true
                ,is_dataflow_pull:false
                ,option:option
            }
        );
        self.Binder_brake=this.CreateBinder({
                binder_name:"Brake"
                ,binder_show_name:"Brake"
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT|DeviceValBinder_TYPE_LIST.VAL_BOOL
                ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_ROBOT^DeviceValBinder_TYPE_LIST.DEVICE_ALL)|DeviceValBinder_TYPE_LIST.VAL_NUMBER
                ,dataflowcb_in:function(val){
                    var cmd= val?"HardStop":"HardBrake"; //true:急停止>ニュートラルモード false:急停止＆ロックモード
                    //モーターへ送信
                    self.Binder_Motor_L.Par_Val=new ST_CtlProtocol(cmd,true);
                    self.Binder_Motor_R.Par_Val=new ST_CtlProtocol(cmd,true);
                }
                ,dataflowcb_get:null
                ,is_dataflow_out:false
                ,is_dataflow_pull:false
                ,option:option
        });
        //--------------------//
        //section::初期化
        //--------------------//
        //--------------------//
        //section::編集用UIに公開するプロパティ
        //--------------------//
        this.PublishUIProp({prop_name:"MaxSpeed",prop_show_name:"Max Speed",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:0,maxlim:300,step:1});
        this.PublishUIProp({prop_name:"DistRatio",prop_show_name:"Turning Ratio",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:0.5,maxlim:4,step:0.1});
        this.PublishUIProp({prop_name:"ReversMotor_L",prop_show_name:"Reverse Left",prop_type:PublishUIProp_TYPE.BOOL,lowlim:null,maxlim:null,step:null});
        this.PublishUIProp({prop_name:"ReversMotor_R",prop_show_name:"Reverse Right",prop_type:PublishUIProp_TYPE.BOOL,lowlim:null,maxlim:null,step:null});
        // -------------------//
        //section::イベントバインド
        //--------------------//
        self.Binder_Motor_L.EventHandler.addListener(self.Binder_Motor_L.EventHandlerType.OnBindChange,function(e){
            //LED情報の取得要求
            self.Binder_Motor_L.Par_Val=new ST_CtlProtocol("GetLedStateColorNotifyOne",true);
        });
        self.Binder_Motor_R.EventHandler.addListener(self.Binder_Motor_R.EventHandlerType.OnBindChange,function(e){
            //LED情報の取得要求
            self.Binder_Motor_R.Par_Val=new ST_CtlProtocol("GetLedStateColorNotifyOne",true);
        });
        ///破棄時のイベント（基底クラスから呼び出される）　
        this.EventHandler.addListener(self.EventHandlerType.OnDestory,function(e){
        });
    }
    //基底クラス継承
    _AbstractDevice_TwoWheelCar_Assembly.prototype = Object.create(_AbstractDevice_module_base.prototype,{constructor:{value:_AbstractDevice_TwoWheelCar_Assembly}});
    // -------------------//
    // section::定数定義
    //--------------------//
    // -------------------//
    // section::パブリックプロパティ
    //--------------------//
    Object.defineProperty(_AbstractDevice_TwoWheelCar_Assembly.prototype,'MaxSpeed', {
        get: function() {
            return this.PROP.MaxSpeed;
        },
        set:function(val){
            if(this.PROP.MaxSpeed===val){return;}
            this.PROP.MaxSpeed=val;
        }
    });
    Object.defineProperty(_AbstractDevice_TwoWheelCar_Assembly.prototype,'DistRatio', {
        get: function() {
            return this.PROP.DistRatio;
        },
        set:function(val){
            this.PROP.DistRatio=val;
        }
    });

    //出力値反転
    Object.defineProperty(_AbstractDevice_TwoWheelCar_Assembly.prototype,'ReversMotor_L', {
        get: function() {
            return this.PROP.ReversMotor_L;
        },
        set:function(val){
            this.PROP.ReversMotor_L=val;
        }
    });
    Object.defineProperty(_AbstractDevice_TwoWheelCar_Assembly.prototype,'ReversMotor_R', {
        get: function() {
            return this.PROP.ReversMotor_R;
        },
        set:function(val){
            this.PROP.ReversMotor_R=val;
        }
    });


    /************************************************************************************
     *  section::***** PanTiltGimbal_Assembly 2軸雲台コントローラー
         ●バインダー
             ◎Control_Vector	(表示名:ControlVector)	//・XY軸の方向ベクトル入力
             ◎Motor_X	(表示名:Motor_X)	//・モーターの制御プロトコル(X軸 モーターに接続)
             ◎Motor_Y	(表示名:Motor_Y)	//・モーターの制御プロトコル(Y軸 モーターに接続)
             ◎RecCtl	(表示名:RecCtl Input)	//・位置を記憶するコントローラーの入力
             ◎Position_free	(表示名:Position_free)	//・モーター停止-->ニュートラル
             ◎RecPosition『Advance』	(表示名:RecPosition)	//・記憶位置を保存するメモリ番号
             ◎PlayPosition『Advance』	(表示名:PlayPosition)	//・記憶位置を再生するメモリ番号
             ◎PossionMemorys『Advance』	(表示名:PossionMemorys)	//・記憶した位置座標の値  null "" undefindeを指定した場合は全消去
             ◎ResetIntegPosition『Advance』	(表示名:Reset IntegPosition)	//・モーターの相対位置をリセットする
             ◎ChangePossionMemorysNotify『Advance』	(表示名:Change PossionMemorys Notify)	//・記憶した位置座標の値 RecPositionで変更の度に送信される
             ◎ClearPossionMemorys『Advance』	(表示名: Clear PossionMemorys)	//・記憶した位置座標の全消去
         ●プロパティ
             ◎ Ratio_X		//・X軸ベクトル入力の移動係数 (最大ベクトル入力時 = 0-360 degree)
             ◎ Ratio_Y		//・Y軸ベクトル入力の移動係数 (最大ベクトル入力時 = 0-360 degree)
             ◎ FollowingSpeed_X		//・X軸の追従速度rpm (記憶位置再生時の移動速度にも使用)
             ◎ FollowingSpeed_Y		//・Y軸の追従速度rpm (記憶位置再生時の移動速度にも使用)
             ◎ ReversMotor_X		//・X軸の回転方向の反転
             ◎ ReversMotor_Y		//・Y軸の回転方向の反転
             ◎ SetRecPosition『Advance』		//・記憶位置を保存するメモリ番号
             ◎ SetPlayPosition『Advance』		//・記憶位置を再生するメモリ番号
             ◎ SetResetPosition『Advance』		//・現在のモーター位置を0として積算位置をリセット
     *
     ************************************************************************************/
    function _AbstractDevice_PanTiltGimbal_Assembly(option,class_option) {
        "use strict";
        _AbstractDevice_module_base.apply(this, arguments);//info::基底クラスが不適切
        var self = this;

        //--------------------//
        //section::追加イベントハンドラ定義
        //--------------------//
        //--------------------//
        //section::内部プロパティ
        //--------------------//
        this._inVector=new ST_Vector2();
        this._Rev_MTRotState_X={};//モーターの状態X notifyに連動
        this._Rev_MTRotState_Y={};//モーターの状態Y
        //前回のモーターのリセット位置
        //this.PROP.PreSetPositon_Motor_x= option.PreSetPositon_Motor_x instanceof Object ?option.PreSetPositon_Motor_x:{};

        this._LedStateColor_X=new ST_MTLedState();//LEDの色 モジュールアイコンSVGの色で使用
        this._LedStateColor_Y=new ST_MTLedState();//LEDの色 モジュールアイコンSVGの色で使用
        //--------------------//
        //section::保存プロパティの復帰＆デフォルト値設定
        //--------------------//
        this.PROP.Ratio_X=option.Ratio_X?option.Ratio_X:360;
        this.PROP.Ratio_Y=option.Ratio_Y?option.Ratio_Y:360;
        this.PROP.FollowingSpeed_X=option.FollowingSpeed_X?option.FollowingSpeed_X:20;
        this.PROP.FollowingSpeed_Y=option.FollowingSpeed_Y?option.FollowingSpeed_Y:20;
        this.PROP.ReversMotor_X=option.ReversMotor_X?true:false;
        this.PROP.ReversMotor_Y=option.ReversMotor_Y?true:false;
        this.PROP.PossionMemorys=option.PossionMemorys?option.PossionMemorys:[];//保存位置

        //--------------------//
        //section::バインダーの生成
        //--------------------//
        self.Binder_Control_Vector=this.CreateBinder(
            {
                binder_name:"Control_Vector"
                ,binder_show_name:'XY Vector'
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT|DeviceValBinder_TYPE_LIST.VAL_VECTOR2
                ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_ROBOT^DeviceValBinder_TYPE_LIST.DEVICE_ALL)|DeviceValBinder_TYPE_LIST.VAL_VECTOR2
                ,dataflowcb_in:function(vector2){
                    self.SetInput=vector2;
                }
                ,dataflowcb_get:null
                ,is_dataflow_out:false
                ,is_dataflow_pull:false
                ,option:option
            }
        );
        self.Binder_Motor_X=this.CreateBinder(
            {
                binder_name:"Motor_X"
                ,binder_show_name:'Motor: Axis X'
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT | DeviceValBinder_TYPE_LIST.VAL_CTL_MT
                ,allow_list:DeviceValBinder_TYPE_LIST.DEVICE_MOTOR | DeviceValBinder_TYPE_LIST.VAL_CTL_MT
                ,dataflowcb_in:function(val){
                    if(val instanceof ST_NotifyProtocol){
                        if(val.Val instanceof ST_MTRotState){
                            //回転情報受信
                            self._Rev_MTRotState_X[val.originID]=val.Val;
                        }else if(val.Val instanceof ST_MTLedState){
                            //LED情報受信
                            self._LedStateColor_X=val.Val;
                            //AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnIconViewLedStateReDrowCall,device:self});//LEDアイコン再描画用
                            AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnIconViewReDrowCall,device:self});//アイコン再描画用
                        }
                    }
                }
                ,dataflowcb_get:null
                ,is_dataflow_out:true
                ,is_dataflow_pull:false
                ,option:option
            }
        );
        self.Binder_Motor_Y=this.CreateBinder(
            {
                binder_name:"Motor_Y"
                ,binder_show_name:'Motor: Axis Y'
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT | DeviceValBinder_TYPE_LIST.VAL_CTL_MT
                ,allow_list:DeviceValBinder_TYPE_LIST.DEVICE_MOTOR | DeviceValBinder_TYPE_LIST.VAL_CTL_MT
                ,dataflowcb_in:function(val){
                    if(val instanceof ST_NotifyProtocol){
                        if(val.Val instanceof ST_MTRotState){
                            //回転情報受信
                            self._Rev_MTRotState_Y[val.originID]=val.Val;
                        }else if(val.Val instanceof ST_MTLedState){
                            //LED情報受信
                            self._LedStateColor_Y=val.Val;
                            //AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnIconViewLedStateReDrowCall,device:self});//LEDアイコン再描画用
                            AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnIconViewReDrowCall,device:self});//アイコン再描画用
                        }
                    }
                }
                ,dataflowcb_get:null
                ,is_dataflow_out:true
                ,is_dataflow_pull:false
                ,option:option
            }
        );
        self.Binder_RecCtl=this.CreateBinder(
            {
                binder_name:"RecCtl"
                ,binder_show_name:"Position Recorder"
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT|DeviceValBinder_TYPE_LIST.VAL_CTL_REC
                ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_ROBOT^DeviceValBinder_TYPE_LIST.DEVICE_ALL)|DeviceValBinder_TYPE_LIST.VAL_CTL_REC
                ,dataflowcb_in:function(val){
                //Commandに対応するプロパティを実行 (info:: UTL_IsSetterOnePerentProto：setterプロパティの動的指定時に、setterと間違えて関数や定数の参照を上書きするのを防止する為に使用)
                if(val instanceof ST_CtlProtocol && UTL_IsSetterOnePerentProto(self,val.Command)){
                    //command指定時にこの値があると、指定したIDの仮想デバイスのみ実行
                    if(val.targetID&&val.targetID!==self.UID){
                        return;
                    }
                    self[val.Command]=val.Val;
                }
            }
                ,dataflowcb_get:function(){
                return new ST_NotifyProtocol("PossionMemorys", self.PossionMemorys, self.UID);
            }
                ,is_dataflow_out:true
                ,is_dataflow_pull:false
                ,option:option
            }
        );
        self.Binder_Position_free=this.CreateBinder(
            {
                binder_name:"Position_free"
                ,binder_show_name:"Position Free"
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT|DeviceValBinder_TYPE_LIST.VAL_BOOL
                ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_ROBOT^DeviceValBinder_TYPE_LIST.DEVICE_ALL)|DeviceValBinder_TYPE_LIST.VAL_NUMBER
                ,dataflowcb_in:function(val){
                    self.SetPositionfree=val;
                }
                ,dataflowcb_get:null
                ,is_dataflow_out:false
                ,is_dataflow_pull:false
                ,option:option
            }
        );

        self.Binder_RecPosition=this.CreateBinder(
            {
                binder_name:"RecPosition"
                ,binder_show_name:"Record Position at Address"
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT|DeviceValBinder_TYPE_LIST.VAL_INT
                ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_ROBOT^DeviceValBinder_TYPE_LIST.DEVICE_ALL)|DeviceValBinder_TYPE_LIST.VAL_INT | DeviceValBinder_TYPE_LIST.VAL_FLOAT
                ,dataflowcb_in:function(val){
                    self.SetRecPosition=val;
                }
                ,dataflowcb_get:null
                ,is_dataflow_out:false
                ,is_dataflow_pull:false
                ,ui_show_type:DeviceValBinder_UI_SHOW_TYPE.Advance
                ,option:option
            }
        );
        self.Binder_PlayPosition=this.CreateBinder(
            {
                binder_name:"PlayPosition"
                ,binder_show_name:"Play Position at Address"
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT|DeviceValBinder_TYPE_LIST.VAL_INT
                ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_ROBOT^DeviceValBinder_TYPE_LIST.DEVICE_ALL)|DeviceValBinder_TYPE_LIST.VAL_INT | DeviceValBinder_TYPE_LIST.VAL_FLOAT
                ,dataflowcb_in:function(val){
                    self.SetPlayPosition=val;
                }
                ,dataflowcb_get:null
                ,is_dataflow_out:false
                ,is_dataflow_pull:false
                ,ui_show_type:DeviceValBinder_UI_SHOW_TYPE.Advance
                ,option:option
            }
        );
        self.Binder_PossionMemorys=this.CreateBinder(
            {
                binder_name:"PossionMemorys"
                ,binder_show_name:"Positions in Memory"
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT|DeviceValBinder_TYPE_LIST.VAL_ARRAY
                ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_ROBOT^DeviceValBinder_TYPE_LIST.DEVICE_ALL)|DeviceValBinder_TYPE_LIST.VAL_ARRAY | DeviceValBinder_TYPE_LIST.VAL_FLOAT | DeviceValBinder_TYPE_LIST.VAL_INT
                ,dataflowcb_in:function(val){self.PossionMemorys=val;}
                ,dataflowcb_get:function(){ return self.PossionMemorys;}
                ,is_dataflow_out:false
                ,is_dataflow_pull:false
                ,ui_show_type:DeviceValBinder_UI_SHOW_TYPE.Advance
                ,option:option
            }
        );
        //info::単なるトリガー 値の設定取得は無効
        //モーターの相対位置をリセットする
        self.Binder_ResetIntegPosition=this.CreateBinder(
            {
                binder_name:"ResetIntegPosition"
                ,binder_show_name:"Reset Position"
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT|DeviceValBinder_TYPE_LIST.VAL_BOOL
                ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_ROBOT^DeviceValBinder_TYPE_LIST.DEVICE_ALL)|DeviceValBinder_TYPE_LIST.VAL_NUMBER
                ,dataflowcb_in:function(val){
                    self.SetResetPosition=val;
                }
                ,dataflowcb_get:null
                ,is_dataflow_out:false
                ,is_dataflow_pull:false
                ,ui_show_type:DeviceValBinder_UI_SHOW_TYPE.Advance
                ,option:option
            }
        );
        self.Binder_Change_PossionMemorys=this.CreateBinder(
            {
                binder_name:"ChangePossionMemorysNotify"
                ,binder_show_name:"Notify Change Position Memory"
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT|DeviceValBinder_TYPE_LIST.VAL_ARRAY
                ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_ROBOT^DeviceValBinder_TYPE_LIST.DEVICE_ALL)|DeviceValBinder_TYPE_LIST.VAL_ARRAY
                ,dataflowcb_in:null
                ,dataflowcb_get:function(){
                    return self.PossionMemorys;//info::Gettrの他にはBinder_RecPositionで変更の度に送信される
                }
                ,is_dataflow_out:true
                ,is_dataflow_pull:false
                ,ui_show_type:DeviceValBinder_UI_SHOW_TYPE.Advance
                ,option:option
            }
        );
        self.Binder_ClearPossionMemorys=this.CreateBinder(
            {
                binder_name:"ClearPossionMemorys"
                ,binder_show_name:"Clear All Position Memory"
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT|DeviceValBinder_TYPE_LIST.VAL_BOOL
                ,allow_list:(DeviceValBinder_TYPE_LIST.DEVICE_ROBOT^DeviceValBinder_TYPE_LIST.DEVICE_ALL)|DeviceValBinder_TYPE_LIST.VAL_NUMBER
                ,dataflowcb_in:function(val){
                    self.PossionMemorys=null;
                //self.SetClearPosition=val;
                }
                ,dataflowcb_get:null
                ,is_dataflow_out:false
                ,is_dataflow_pull:false
                ,ui_show_type:DeviceValBinder_UI_SHOW_TYPE.Advance
                ,option:option
            }
        );


        //--------------------//
        //section::初期化
        //--------------------//

        //--------------------//
        //section::編集用UIに公開するプロパティ
        //--------------------//
        this.PublishUIProp({prop_name:"Ratio_X",prop_show_name:"Ratio X",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:0,maxlim:360,step:45});
        this.PublishUIProp({prop_name:"Ratio_Y",prop_show_name:"Ratio Y",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:0,maxlim:360,step:45});
        this.PublishUIProp({prop_name:"FollowingSpeed_X",prop_show_name:"FollowingSpeed X",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:0,maxlim:200,step:20});//rpm
        this.PublishUIProp({prop_name:"FollowingSpeed_Y",prop_show_name:"FollowingSpeed Y",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:0,maxlim:200,step:20});
        this.PublishUIProp({prop_name:"ReversMotor_X",prop_show_name:"Reverse X",prop_type:PublishUIProp_TYPE.BOOL,lowlim:null,maxlim:null,step:null});
        this.PublishUIProp({prop_name:"ReversMotor_Y",prop_show_name:"Reverse Y",prop_type:PublishUIProp_TYPE.BOOL,lowlim:null,maxlim:null,step:null});
        this.PublishUIProp({prop_name:"SetRecPosition",prop_show_name:"Record Position at Address",prop_type:PublishUIProp_TYPE.INT,lowlim:null,maxlim:null,step:null,ui_show_type:PublishUIProp_UI_SHOW_TYPE.Advance});
        this.PublishUIProp({prop_name:"SetPlayPosition",prop_show_name:"Play Position at Address",prop_type:PublishUIProp_TYPE.INT,lowlim:null,maxlim:null,step:null,ui_show_type:PublishUIProp_UI_SHOW_TYPE.Advance});
        this.PublishUIProp({prop_name:"SetResetPosition",prop_show_name:"Reset Position",prop_type:PublishUIProp_TYPE.SEND_NULL,lowlim:null,maxlim:null,step:null,ui_show_type:PublishUIProp_UI_SHOW_TYPE.Advance});
        // -------------------//
        //section::イベントバインド
        //--------------------//
        //バインド切り替え時
        self.Binder_Motor_X.EventHandler.addListener(self.Binder_Motor_X.EventHandlerType.OnBindChange,function(e){
            //LED情報の取得要求
            self.Binder_Motor_X.Par_Val=new ST_CtlProtocol("GetLedStateColorNotifyOne",true);
        });
        //バインド切り替え時
        self.Binder_Motor_Y.EventHandler.addListener(self.Binder_Motor_Y.EventHandlerType.OnBindChange,function(e){
            //LED情報の取得要求
            self.Binder_Motor_Y.Par_Val=new ST_CtlProtocol("GetLedStateColorNotifyOne",true);
        });
        ///破棄時のイベント（基底クラスから呼び出される）　
        this.EventHandler.addListener(self.EventHandlerType.OnDestory,function(e){
        });
    }
    //基底クラス継承
    _AbstractDevice_PanTiltGimbal_Assembly.prototype = Object.create(_AbstractDevice_module_base.prototype,{constructor:{value:_AbstractDevice_PanTiltGimbal_Assembly}});
    // -------------------//
    // section::定数定義
    //--------------------//
    // -------------------//
    // section::パブリックプロパティ
    //--------------------//
    Object.defineProperty(_AbstractDevice_PanTiltGimbal_Assembly.prototype,'Ratio_X', {
        get: function() {
            return this.PROP.Ratio_X;
        },
        set:function(val){
            this.PROP.Ratio_X=val;
        }
    });
    //
    Object.defineProperty(_AbstractDevice_PanTiltGimbal_Assembly.prototype,'Ratio_Y', {
        get: function() {
            return this.PROP.Ratio_Y;
        },
        set:function(val){
            this.PROP.Ratio_Y=val;
        }
    });
    Object.defineProperty(_AbstractDevice_PanTiltGimbal_Assembly.prototype,'FollowingSpeed_X', {
        get: function() {
            return this.PROP.FollowingSpeed_X;
        },
        set:function(val){
            this.PROP.FollowingSpeed_X=val;
        }
    });
    //
    Object.defineProperty(_AbstractDevice_PanTiltGimbal_Assembly.prototype,'FollowingSpeed_Y', {
        get: function() {
            return this.PROP.FollowingSpeed_Y;
        },
        set:function(val){
            this.PROP.FollowingSpeed_Y=val;
        }
    });
    //出力値反転
    Object.defineProperty(_AbstractDevice_PanTiltGimbal_Assembly.prototype,'ReversMotor_X', {
        get: function() {
            return this.PROP.ReversMotor_X;
        },
        set:function(val){
            this.PROP.ReversMotor_X=val;
        }
    });
    Object.defineProperty(_AbstractDevice_PanTiltGimbal_Assembly.prototype,'ReversMotor_Y', {
        get: function() {
            return this.PROP.ReversMotor_Y;
        },
        set:function(val){
            this.PROP.ReversMotor_Y=val;
        }
    });
    //入力されたベクトル値をモーターに送る vector2
    Object.defineProperty(_AbstractDevice_PanTiltGimbal_Assembly.prototype,'SetInput', {
        get: function() {
            return null;
        },
        set:function(vector2){
            if(! (vector2 instanceof ST_Vector2)){return;}
            var _nowVector=vector2.Clone();
            // console.dir(vector2);
            //0,0が来た場合のリセット(タッチアップ時)
            if(_nowVector.x===0&&_nowVector.y===0){
                this._inVector=_nowVector;
                return;
            }
            //ベクトル値による分配
            var _x=(_nowVector.x-this._inVector.x)*(this.Ratio_X/2);
            var _y=(_nowVector.y-this._inVector.y)*(this.Ratio_Y/2);
            // console.log(" x:"+vector2.x+" y:"+vector2.y+" _x:"+_x+" _y:"+_y);

            var _X_deg=this.ReversMotor_X?_x*-1:_x;
            var _Y_deg=this.ReversMotor_Y?_y*-1:_y;
            //モーターへ送信
            this.Binder_Motor_X.Par_Val=new ST_CtlProtocol("SetRelPosition_deg",[_X_deg,this.FollowingSpeed_X,0]);
            this.Binder_Motor_Y.Par_Val=new ST_CtlProtocol("SetRelPosition_deg",[_Y_deg,this.FollowingSpeed_Y,0]);

            this._inVector=_nowVector;

        }
    });

    Object.defineProperty(_AbstractDevice_PanTiltGimbal_Assembly.prototype,'SetPositionfree', {
        get: function() {
            return null;
        },
        set:function(val){
            //モーターへ送信
            this.Binder_Motor_X.Par_Val=new ST_CtlProtocol("HardStop",true);
            this.Binder_Motor_Y.Par_Val=new ST_CtlProtocol("HardStop",true);
        }
    });
    //現在の位置を0として積算位置をリセット
    Object.defineProperty(_AbstractDevice_PanTiltGimbal_Assembly.prototype,'SetResetPosition', {
        get: function() {
            return null;
        },
        set:function(val){
            if(val){
                this._inVector=new ST_Vector2();
                //モーターの絶対位置をリセット
                // var key=Object.keys(this._Rev_MTRotState_X);
                // for(var i=0;i<key.length;i++) {
                //     this.PROP.PreSetPositon_Motor_x[key[i]]=0;
                // }
                // var key=Object.keys(this._Rev_MTRotState_Y);
                // for(var i=0;i<key.length;i++) {
                //     this.PROP.PreSetPositon_Motor_y[key[i]]=0;
                // }
                this.Binder_Motor_X.Par_Val=new ST_CtlProtocol("PreSetPositon",0);
                this.Binder_Motor_Y.Par_Val=new ST_CtlProtocol("PreSetPositon",0);
            }
        }
    });
    //SetRecPosition(W):int[0 - n] 記憶位置を保存するメモリ番号
    Object.defineProperty(_AbstractDevice_PanTiltGimbal_Assembly.prototype,'SetRecPosition', {
        get: function() {return null;
        },
        set:function(val){
            var x_pos={};
            var key=Object.keys(this._Rev_MTRotState_X);
            for(var i=0;i<key.length;i++) {
                var k=key[i];
                x_pos[k]=UTL_N(this._Rev_MTRotState_X[k].Position);
            }
            var y_pos={};
            var key=Object.keys(this._Rev_MTRotState_Y);
            for(var i=0;i<key.length;i++) {
                var k=key[i];
                y_pos[k]=UTL_N(this._Rev_MTRotState_Y[k].Position);
            }

            this.PROP.PossionMemorys[UTL_INT(val)]={x:x_pos,y:y_pos};
            console.log("set PossionMemorys["+UTL_INT(val)+"]={x:"+x_pos+",y:"+y_pos+"}");
            this.Binder_RecCtl.Par_Val=new ST_NotifyProtocol("PossionMemorys",this.PossionMemorys,this.UID);//記憶情報の送信
            this.Binder_Change_PossionMemorys.Par_Val=this.PossionMemorys;
        }

    });
    //SetPlayPosition(W):int[0 - 4]記憶位置を再生するメモリ番号
    Object.defineProperty(_AbstractDevice_PanTiltGimbal_Assembly.prototype,'SetPlayPosition', {
        get: function() {
            return null;
        },
        set:function(val){
            var obj=this.PROP.PossionMemorys[UTL_INT(val)];
            UTL_LOG("play PossionMemorys["+UTL_INT(val)+"]={x:"+obj.x+",y:"+obj.y+"}");
            if(obj){
                this._SetMorterXYPosition(obj.x,obj.y);
            }
        }
    });

    //SetClearPosition(W):int[0 - 4]記憶位置を消去するメモリ番号 undefinde nullを指定した場合は全消去
    Object.defineProperty(_AbstractDevice_PanTiltGimbal_Assembly.prototype,'SetClearPosition', {
        get: function() {
            return null;
        },
        set:function(val){
            if(val==null){
                this.PossionMemorys=null;
            }else{
                this.PROP.PossionMemorys[UTL_INT(val)]=null;
                this.Binder_RecCtl.Par_Val=new ST_NotifyProtocol("PossionMemorys",this.PossionMemorys,this.UID);//記憶情報の送信
                this.Binder_Change_PossionMemorys.Par_Val=this.PossionMemorys;
           }
        }
    });

    //PossionMemorys(RW)
    Object.defineProperty(_AbstractDevice_PanTiltGimbal_Assembly.prototype,'PossionMemorys', {
        get: function() {
            //PossionMemorysの複製を返す
            var mem=this.PROP.PossionMemorys;
            var rt=[];
            for(var i=0;i<mem.length;i++){
                if(mem[i] instanceof Object){
                    rt[i]=UTL_clone(mem[i]);
                }
            }
            return rt;
        },
        set:function(val){
            this.PROP.PossionMemorys=[];
            if(val instanceof Array){
                for(var i=0;i=val.length;i++){
                    this.PROP.PossionMemorys[i]=val[i];
                }
            }
            this.Binder_RecCtl.Par_Val=new ST_NotifyProtocol("PossionMemorys",this.PossionMemorys,this.UID);//記憶情報の送信
            this.Binder_Change_PossionMemorys.Par_Val=this.PossionMemorys;
        }
    });


    // -------------------//
    // section::内部関数
    //--------------------//

    //radianでモーターのXY位置を設定する（メモリ設定用）
    _AbstractDevice_PanTiltGimbal_Assembly.prototype._SetMorterXYPosition=function(x_radians,y_radians){
        if(x_radians instanceof Object) {
            var key=Object.keys(x_radians);
            var speed=this.FollowingSpeed_X;
            for(var i=0;i<key.length;i++) {
                this.Binder_Motor_X.Par_Val = new ST_CtlProtocol("SetPosition", new ST_MTRotState(x_radians[key[i]], UTL_conv_rpm_to_radi_sec(speed), ST_MTRotState.MAX_TORQUE),key[i]);//FollowingSpeed_X:RPM ->  (radians/second) に
            }
        }
        if(y_radians instanceof Object) {
            var key=Object.keys(y_radians);
            var speed=this.FollowingSpeed_Y;
            for(var i=0;i<key.length;i++) {
                this.Binder_Motor_Y.Par_Val = new ST_CtlProtocol("SetPosition", new ST_MTRotState(y_radians[key[i]], UTL_conv_rpm_to_radi_sec(speed), ST_MTRotState.MAX_TORQUE),key[i]);//FollowingSpeed_Y:RPM ->  (radians/second) に
            }
        }
    };

    /************************************************************************************
     *  section::***** SensingTurnTable_Assembly ターンテーブル(万華鏡用)
         ●バインダー
             ◎MotorIN	(表示名:INPUT Motor)	//・入力(センシング操作用)モーターに接続
             ◎MotorOUT	(表示名:OUTPUT Motor)	//・出力(ターンテーブル用)モーターに接続
         ●プロパティ
             ◎ AutoMode		//・AutoModeの有効・無効
             ◎ AutoModeSpeedRPM		//・自動で回転する速度 rpm
             ◎ AutoModeThreshold		//・入力モータの回転速度がこの値以下になるとAutoModeになる値 rpm
             ◎ AutoModeIntervalTime		//・AutoModeThresholdRPM以下になってからAutoModeになるまでのインターバル msec
             ◎ InertiaLowLimitSpeed		//・入力モーターの位置を受けて回転する出力モーターの下限となるスピード rpm
             ◎ MotorSwap		//・入力と出力モーターを入れ替え
     ************************************************************************************/
    function _AbstractDevice_SensingTurnTable_Assembly(option,class_option) {
        "use strict";
        _AbstractDevice_module_base.apply(this, arguments);//info::基底クラスが不適切
        var self = this;

        //--------------------//
        //section::追加イベントハンドラ定義
        //--------------------//
        //--------------------//
        //section::内部プロパティ
        //--------------------//

        this.m_to={};//モーターの参照　MotorSwap用
        this.m_from={};
        this.State_to=null;
        this.State_from=null;
        this.Rev_MTRotState_to=null;
        this.Rev_MTRotState_from=null;

        this._SyncMotorPos=0;//入力モーターのリセットした位置
        this._IntervalCount=0;
        this._nowsp=0;
        this._IsAutoModePlaying=false;
        this._Rev_MTRotState_IN={RotState:new ST_MTRotState()};
        this._Rev_MTRotState_OUT={RotState:new ST_MTRotState()};
        self._LedStateColor_IN=new ST_MTLedState();
        self._LedStateColor_OUT=new ST_MTLedState();
        //--------------------//
        //section::保存プロパティの復帰＆デフォルト値設定
        //--------------------//
        this.PROP.AutoModeSpeed=option.AutoModeSpeed?option.AutoModeSpeed:-1;
        this.PROP.AutoModeThreshold=option.AutoModeThreshold?option.AutoModeThreshold:5;
        this.PROP.AutoModeIntervalTime=option.AutoModeIntervalTime?option.AutoModeIntervalTime:1000;
        this.PROP.AutoMode=(typeof option.AutoMode==='boolean')?option.AutoMode:true;//info::default true
        this.PROP.MotorSwap=option.MotorSwap?true:false;

        this.PROP.InertiaLowLimitSpeed=option.InertiaLowLimitSpeed?option.InertiaLowLimitSpeed:1;

        //--------------------//
        //section::バインダーの生成
        //--------------------//
        this.MotorIN=this.CreateBinder(
            {
                binder_name:"MotorIN"
                ,binder_show_name:'Motor: In'
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT | DeviceValBinder_TYPE_LIST.VAL_CTL_MT
                ,allow_list:DeviceValBinder_TYPE_LIST.DEVICE_MOTOR | DeviceValBinder_TYPE_LIST.VAL_CTL_MT
                ,dataflowcb_in:function(val){
                    //info::回転情報受信
                    if(val instanceof ST_NotifyProtocol){
                        if(val.Val instanceof ST_MTRotState){
                            //回転情報受信
                            self._Rev_MTRotState_IN.RotState=val.Val;
                            self.State_from(val.Val);
                        }else if(val.Val instanceof ST_MTLedState){
                            //LED情報受信
                            self._LedStateColor_IN=val.Val;
                           //AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnIconViewLedStateReDrowCall,device:self});//LEDアイコン再描画用
                            AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnIconViewReDrowCall,device:self});//アイコン再描画用
                        }
                    }
                }
                ,dataflowcb_get:null
                ,is_dataflow_out:true
                ,is_dataflow_pull:false
                ,option:option
            }
        );

        this.MotorOUT=this.CreateBinder(
            {
                binder_name:"MotorOUT"
                ,binder_show_name:'Motor: Out'
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT | DeviceValBinder_TYPE_LIST.VAL_CTL_MT
                ,allow_list:DeviceValBinder_TYPE_LIST.DEVICE_MOTOR | DeviceValBinder_TYPE_LIST.VAL_CTL_MT
                ,dataflowcb_in:function(val){
                    //info::回転情報受信
                    if(val instanceof ST_NotifyProtocol){
                        if(val.Val instanceof ST_MTRotState){
                            //回転情報受信
                            self._Rev_MTRotState_OUT.RotState=val.Val;
                            self.State_to(val.Val);
                        }else if(val.Val instanceof ST_MTLedState){
                            //LED情報受信
                            self._LedStateColor_OUT=val.Val;
                           // AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnIconViewLedStateReDrowCall,device:self});//LEDアイコン再描画用
                            AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnIconViewReDrowCall,device:self});//アイコン再描画用
                        }
                    }
                }
                ,dataflowcb_get:null
                ,is_dataflow_out:true
                ,is_dataflow_pull:false
                ,option:option
            }
        );

        //--------------------//
        //section::初期化
        //--------------------//

        this.MotorSwap=this.PROP.MotorSwap;//入れ替えの初期化
        //--------------------//
        //section::編集用UIに公開するプロパティ
        //--------------------//
        this.PublishUIProp({prop_name:"AutoMode",prop_show_name:"Enable Auto Rotation",prop_type:PublishUIProp_TYPE.BOOL,lowlim:null,maxlim:null,step:true});
        this.PublishUIProp({prop_name:"AutoModeSpeed",prop_show_name:"Auto Rotation Speed (rpm)",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:-200,maxlim:200,step:1});
        this.PublishUIProp({prop_name:"AutoModeThreshold",prop_show_name:"Auto Rotation Threshold (rpm)",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:5,maxlim:50,step:5});//info::5rpm以下だとエンコーダーのノイズに反応し誤動作する
        this.PublishUIProp({prop_name:"AutoModeIntervalTime",prop_show_name:"Auto Rotation Exec. Time",prop_type:PublishUIProp_TYPE.INT,lowlim:10,maxlim:10000,step:1000});
        this.PublishUIProp({prop_name:"InertiaLowLimitSpeed",prop_show_name:"Output Rotation Min Speed (rpm)",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:0,maxlim:10,step:1});
        this.PublishUIProp({prop_name:"MotorSwap",prop_show_name:"Swap In and Out",prop_type:PublishUIProp_TYPE.BOOL,lowlim:null,maxlim:null,step:null});
        // -------------------//
        //section::イベントバインド
        //--------------------//
        self.MotorOUT.EventHandler.addListener(self.MotorOUT.EventHandlerType.OnBindChange,function(e){
            //LED情報の取得要求
            self.MotorOUT.Par_Val=new ST_CtlProtocol("GetLedStateColorNotifyOne",true);
        });
        self.MotorIN.EventHandler.addListener(self.MotorIN.EventHandlerType.OnBindChange,function(e){
            //LED情報の取得要求
            self.MotorIN.Par_Val=new ST_CtlProtocol("GetLedStateColorNotifyOne",true);
        });
        ///破棄時のイベント（基底クラスから呼び出される）　
        this.EventHandler.addListener(self.EventHandlerType.OnDestory,function(e){
        });
    }
    //基底クラス継承
    _AbstractDevice_SensingTurnTable_Assembly.prototype = Object.create(_AbstractDevice_module_base.prototype,{constructor:{value:_AbstractDevice_SensingTurnTable_Assembly}});
    // -------------------//
    // section::定数定義
    //--------------------//
    // -------------------//
    // section::パブリックプロパティ
    //--------------------//
    Object.defineProperty(_AbstractDevice_SensingTurnTable_Assembly.prototype,'AutoModeSpeed', {
        get: function() {
            return this.PROP.AutoModeSpeed;
        },
        set:function(val){
            this.PROP.AutoModeSpeed=val;
        }
    });
    Object.defineProperty(_AbstractDevice_SensingTurnTable_Assembly.prototype,'AutoModeThreshold', {
        get: function() {
            return this.PROP.AutoModeThreshold;
        },
        set:function(val){
            this.PROP.AutoModeThreshold=val;
        }
    });
    Object.defineProperty(_AbstractDevice_SensingTurnTable_Assembly.prototype,'AutoModeIntervalTime', {
        get: function() {
            return this.PROP.AutoModeIntervalTime;
        },
        set:function(val){
            this.PROP.AutoModeIntervalTime=val;
        }
    });
    Object.defineProperty(_AbstractDevice_SensingTurnTable_Assembly.prototype,'InertiaLowLimitSpeed', {
        get: function() {
            return this.PROP.InertiaLowLimitSpeed;
        },
        set:function(val){
            this.PROP.InertiaLowLimitSpeed=val;
        }
    });
    Object.defineProperty(_AbstractDevice_SensingTurnTable_Assembly.prototype,'AutoMode', {
        get: function() {
            return this.PROP.AutoMode;
        },
        set:function(val){
            this.PROP.AutoMode=val;
            this._SetMotorAutoPlay(val);
        }
    });
    Object.defineProperty(_AbstractDevice_SensingTurnTable_Assembly.prototype,'MotorSwap', {
        get: function() {
            return this.PROP.MotorSwap;
        },
        set:function(val){
            if(val){
                this.m_from=this.MotorOUT;
                this.m_to=this.MotorIN;
                this.State_from=this._MotorOUTRotationStateChange;
                this.State_to=this._MotorINRotationStateChange;
                this.Rev_MTRotState_from=this._Rev_MTRotState_OUT;
                this.Rev_MTRotState_to=this._Rev_MTRotState_IN;
            }else{
                this.m_from=this.MotorIN;
                this.m_to=this.MotorOUT;
                this.State_from=this._MotorINRotationStateChange;
                this.State_to=this._MotorOUTRotationStateChange;
                this.Rev_MTRotState_from=this._Rev_MTRotState_IN;
                this.Rev_MTRotState_to=this._Rev_MTRotState_OUT;
            }
            this.PROP.MotorSwap=val?true:false;
            this._ReSyncMotorPosion();
            this._SetMotorAutoPlay(false);
        }
    });
    // -------------------//
    // section::内部関数
    //--------------------//
    //バインドしたモーターの回転情報受信リスナー
    _AbstractDevice_SensingTurnTable_Assembly.prototype._MotorINRotationStateChange=function(st_MTRotstate){
        var speed=(Math.abs(st_MTRotstate.Speed)+this._nowsp)/4;//速度の入力値を滑らかにする平均化フィルター
        var set_p_speed=speed<this.InertiaLowLimitSpeed?this.InertiaLowLimitSpeed:speed;
        var d = new Date();var ms = d.getTime();

        if(UTL_conv_rpm_to_radi_sec(this.AutoModeThreshold) < speed){
            // console.log("制御モード");
            if(this._IsAutoModePlaying){
                //AUTOモード->制御モードの場合にはモーターposionをリセットする
                this._ReSyncMotorPosion();
            }
            //todo::万華鏡モジュール、回転がぎこちない＞ROTSTATEのバグ？
            this.m_to.Par_Val=new ST_CtlProtocol("SetPosition",new ST_MTRotState(st_MTRotstate.Position+this._SyncMotorPos, set_p_speed, ST_MTRotState.MAX_TORQUE));
            this._IntervalCount=ms;//タイマーリセット
            this._IsAutoModePlaying=false;
            this._nowsp=speed;
        }else{
            //AutoModeモードインターバルカウント
            if(this._IsAutoModePlaying||!this.AutoMode){return;}
            // console.log("AutoModeモードインターバルカウント");
            if(this.AutoModeIntervalTime<(ms-this._IntervalCount)){
                this._SetMotorAutoPlay(true);
            }
        }
    };
    _AbstractDevice_SensingTurnTable_Assembly.prototype._MotorOUTRotationStateChange=function(st_MTRotstate){

    };

    /**
     * モーターのAUTOモード再生
     * @param bool 再生/停止
     * @private
     */
    _AbstractDevice_SensingTurnTable_Assembly.prototype._SetMotorAutoPlay=function(bool){
        this._IsAutoModePlaying=bool;
        if(bool){
            //console.log("_SetMotorAutoPlay PLAY");
            this.m_from.Par_Val=new ST_CtlProtocol("HardStop",true);
            this.m_to.Par_Val=new ST_CtlProtocol("SetPosition",new ST_MTRotState(this.AutoModeSpeed<0?Number.MIN_SAFE_INTEGER:Number.MAX_SAFE_INTEGER,UTL_conv_rpm_to_radi_sec(Math.abs(this.AutoModeSpeed)),ST_MTRotState.MAX_TORQUE));
        }else{
            // console.log("_SetMotorAutoPlay stop");
            this.m_from.Par_Val=new ST_CtlProtocol("HardStop",true);
            this.m_to.Par_Val=new ST_CtlProtocol("HardStop",true);
        }
    };
    _AbstractDevice_SensingTurnTable_Assembly.prototype._ReSyncMotorPosion=function(){
       // console.log("_ReSyncMotorPosion");
        this._SyncMotorPos=this.Rev_MTRotState_to.RotState.Position-this.Rev_MTRotState_from.RotState.Position;
        if(this._SyncMotorPos>10000||this._SyncMotorPos<-10000){
            console.log("err");
        }
    };

    /************************************************************************************
     *  section::***** MotorTransmission_Assembly モーター変速機(ギヤーボックス)　　
     *   モーターの回転速度・座標を 1: GearRatioで変速する
             ●バインダー
                 ◎IN	(表示名:INPUT Motor Control)	//・変速前のモーター制御プロトコル(モジュール側に接続)
                 ◎OUT	(表示名:OUTPUT Motor Control)	//・変速後のモーター制御プロトコル(モーター側に接続)
             ●プロパティ
                 ◎ GearRatio		//・変速比 1: GearRatio
     ************************************************************************************/
    function _AbstractDevice_MotorTransmission_Assembly(option,class_option) {
        "use strict";
        _AbstractDevice_module_base.apply(this, arguments);
        var self = this;
        //--------------------//
        //section::追加イベント
        //--------------------//
        //--------------------//
        //section::内部プロパティ
        //--------------------//
        this._SetLinkPositionFilterSpeed=0;
        this._IsRotationStateUpdated=false;//回転情報が1度でも実機から更新されたか
        this._SetLinkPosition_integ_val=null;//モーター連動開始時のモーターの回転座標
        this._smpl_interval_msec=500;//遅延限界時間 ms
        this._Position=0;
        this._Speed=0;
        this._Torque=0;
        //--------------------//
        //section::保存プロパティの復帰＆デフォルト値設定
        this.PROP.GearRatio= option.GearRatio?option.GearRatio:1;
        //--------------------//
        //--------------------//
        //section::バインダーの生成
        //--------------------//

        //バインダーIN
        self.Bind_IN=this.CreateBinder(
            {
                binder_name:"IN"
                ,binder_show_name:'In'
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_MOTOR|DeviceValBinder_TYPE_LIST.DEVICE_ROBOT | DeviceValBinder_TYPE_LIST.VAL_CTL_MT|DeviceValBinder_TYPE_LIST.VAL_MT_ROTSTATE
                ,allow_list: DeviceValBinder_TYPE_LIST.DEVICE_ALL|DeviceValBinder_TYPE_LIST.VAL_CTL_MT|DeviceValBinder_TYPE_LIST.VAL_MT_ROTSTATE
                ,dataflowcb_in:function(val){
                    //info::回転情報受信
                    if(val instanceof ST_MTRotState){//Binder_RotationStateNotifyバインダーからの値
                        self._SetLinkPosition(val);
                    }
                    else if(val instanceof ST_NotifyProtocol && val.Val instanceof ST_MTRotState){ //ST_MTRotState経由のnotify(ST_MTRotState)
                        self._SetLinkPosition(val.Val);
                    }else if(val instanceof ST_CtlProtocol ){ //制御コマンド
                        //位置制御のみ変速を適応
                        switch (val.Command){
                            case "SetPosition":case "SetPosition_deg":case "SetRelPosition":case "SetRelPosition_deg":
                                //todo::変速処理
                                self.Bind_OUT.Par_Val=new ST_CtlProtocol(val.Command,ST_MTRotState.ConvertGearRatio(val.Val,self.GearRatio),val.targetID);
                                break;
                            case "SetSpeed":case "SetSpeed_rpm"://速度設定
                                //todo::変速処理
                                self.Bind_OUT.Par_Val=new ST_CtlProtocol(val.Command,val.Val*self.GearRatio,val.targetID);
                                break;
                            default :
                                self.Bind_OUT.Par_Val= val;
                                break;
                        }
                    }
                }
                ,dataflowcb_get:null
                ,is_dataflow_out:true
                ,is_dataflow_pull:false
                ,option:option
            }
        );
        //バインド切り替え時
        self.Bind_IN.EventHandler.addListener(self.Bind_IN.EventHandlerType.OnBindChange,function(e){
            self._SetLinkPosition_integ_val=null;
            self._IsRotationStateUpdated=false;
        });

        //バインダーOUT
        self.Bind_OUT=this.CreateBinder(
            {
                binder_name:"OUT"
                ,binder_show_name:"Out"
                ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_ROBOT | DeviceValBinder_TYPE_LIST.VAL_CTL_MT
                ,allow_list:DeviceValBinder_TYPE_LIST.DEVICE_MOTOR | DeviceValBinder_TYPE_LIST.VAL_CTL_MT
                ,dataflowcb_in:function(val){
                    //info::回転情報受信
                    if(val instanceof ST_NotifyProtocol){
                        if(val.Val instanceof ST_MTRotState){
                            var redu_rst=val.Val;
                            self._Position=redu_rst.Position;
                            self._Speed=redu_rst.Speed;
                            self._Torque=redu_rst.Torque;
                            self._IsRotationStateUpdated=true;

                            redu_rst.ConvertGearRatio(1/self.GearRatio);//変速比変換して返す 破壊

                            self.Bind_IN.Par_Val=val;

                        }else if(val.NotifyName===self.EventHandlerType.OnStatusChange){
                            //モーターの相対位置をリセット
                            self._SetLinkPosition_integ_val=null;
                            self._IsRotationStateUpdated=false;
                            self.Bind_IN.Par_Val=val;
                        }else{
                            self.Bind_IN.Par_Val=val;
                        }
                    }
                }
                ,dataflowcb_get:null
                ,is_dataflow_out:true
                ,is_dataflow_pull:false
                ,option:option
            }
        );
        //バインド切り替え時
        self.Bind_OUT.EventHandler.addListener(self.Bind_OUT.EventHandlerType.OnBindChange,function(e){
            self._SetLinkPosition_integ_val=null;
            self._IsRotationStateUpdated=false;
        });

        //--------------------//
        //  初期化
        //--------------------//
        //--------------------//
        //  編集用UIに公開するプロパティ
        //--------------------//
        this.PublishUIProp({prop_name:"GearRatio",prop_show_name:"Gear Ratio",prop_type:PublishUIProp_TYPE.INT,lowlim:1,maxlim:10,step:1});
        // -------------------//
        //  イベントバインド
        //--------------------//
        ///破棄時のイベント（基底クラスから呼び出される）　
        this.EventHandler.addListener(self.EventHandlerType.OnDestory,function(e){});
        ////

    }
    //基底クラス継承
    _AbstractDevice_MotorTransmission_Assembly.prototype = Object.create(_AbstractDevice_module_base.prototype,{constructor:{value:_AbstractDevice_MotorTransmission_Assembly}});
    // -------------------//
    // section::パブリックプロパティ
    //--------------------//
    Object.defineProperty(_AbstractDevice_MotorTransmission_Assembly.prototype,'GearRatio', {
        get: function() {
            return this.PROP.GearRatio;
        },
        set:function(val){
            this.PROP.GearRatio=val;
            this._SetLinkPosition_integ_val=null;
            this._IsRotationStateUpdated=false;
        }
    });
    // -------------------//
    // section::内部関数
    //--------------------//
    //モーター連動　入力された回転情報にモーターを追従し他のモーターと連動 他のモーターのnotify(ST_MTRotState)を接続。
    _AbstractDevice_MotorTransmission_Assembly.prototype._SetLinkPosition=function(_mt_rot_state){
        var mt_rot_state=_mt_rot_state.Clone().ConvertGearRatio(this.GearRatio);
        if(!this._IsRotationStateUpdated){return;}
        //notifyの積算値より平滑化した値で滑らかに連動する
        //親モータと自モーターの位置の差分計算
        if(this._SetLinkPosition_integ_val==null ){
            this._SetLinkPosition_integ_val=mt_rot_state.Position-this._Position;
        }

        var tar_pos=mt_rot_state.Position-this._SetLinkPosition_integ_val;
        var discrepancy_pos=tar_pos-this._Position;//目標位置との距離の差分

       if(Math.abs(discrepancy_pos)<0.1){return;}//同期下限リミット

        var offset_sp=Math.abs(discrepancy_pos/(this._smpl_interval_msec/1000));//遅延限界時間までに、追従するために不足している速度 rad/sec
        var now_sp=this._SetLinkPositionFilterSpeed;
        var torque=mt_rot_state.Torque;
        //一定の開きがある間は以前と現在で速い方のspeed値を維持
        var tar_sp= offset_sp>now_sp?Math.max(Math.abs(mt_rot_state.Speed),this._SetLinkPositionFilterSpeed):Math.abs(mt_rot_state.Speed);
        var speed=(tar_sp+this._SetLinkPositionFilterSpeed)*0.5;//速度の入力値を滑らかにする平均化フィルター
        speed=speed<this.GearRatio?this.GearRatio:speed;//最低リミット速度
        this._SetLinkPositionFilterSpeed=speed;

        var rotst=new ST_MTRotState(tar_pos,speed,torque);
        this.Bind_OUT.Par_Val=new ST_CtlProtocol("SetPosition", rotst);
        //console.dir(rotst);
        //console.log("tar_pos:"+tar_pos);
        //console.table({Position:mt_rot_state.Position,discrepancy_pos:discrepancy_pos,Speed:speed,offset_sp:offset_sp,now_sp:now_sp});
    };



    /************************************************************************************
     *  section::***** Debug_Trace デバッグトレース
         ●バインダー
             ◎IN	(表示名:Send Data)	//・データ入力側
             ◎OUT	(表示名:Return Data)	//・データ出力側
         ●プロパティ
             ◎ Active		//・トレース出力のON/OFF (OFFにすると負荷軽減)
             ◎ Log		//・トレース内容
     ************************************************************************************/
    function _AbstractDevice_Debug_Trace(option,class_option) {
        "use strict";
        _AbstractDevice_module_base.apply(this, arguments);
        var self = this;
        //--------------------//
        //section::追加イベント
        //--------------------//
        this.EventHandlerType["OnDataChange"]="OnDataChange";
        //--------------------//
        //section::内部プロパティ
        //--------------------//
        this._debug_str=">_";
        this._OnIconViewReDrowCallInterval=1000;//デバッグ値のアイコン表示のリフレッシュ間隔1秒
        this._OnIconViewReDrowCallInterval_stacktime=0;
        //--------------------//
        //section::保存プロパティの復帰＆デフォルト値設定
        this.PROP.Active=option.Active===undefined?true:Boolean(option.Active);//default true
        //--------------------//
        //--------------------//
        //section::バインダーの生成
        //--------------------//

        //バインダーIN
        self.Bind_IN=this.CreateBinder(
            {
                binder_name:"IN"
                ,binder_show_name:'In'
                ,type_list:DeviceValBinder_TYPE_LIST.VAL_ALL
                ,allow_list:DeviceValBinder_TYPE_LIST.VAL_ALL
                ,dataflowcb_in:function(val){
                    if(self.Active){
                        self._extract_trace_data(val);
                    }
                    self.Bind_OUT.Par_Val=val;
                }
                ,dataflowcb_get:null
                ,is_dataflow_out:true
                ,is_dataflow_pull:false
                ,option:option
            }
        );
        //バインド切り替え時
        self.Bind_IN.EventHandler.addListener(self.Bind_IN.EventHandlerType.OnBindChange,function(e){
            self._bindcheck();
        });

        //バインダーOUT
        self.Bind_OUT=this.CreateBinder(
            {
                binder_name:"OUT"
                ,binder_show_name:'Out'
                ,type_list: DeviceValBinder_TYPE_LIST.VAL_ALL
                ,allow_list:DeviceValBinder_TYPE_LIST.VAL_ALL
                ,dataflowcb_in:function(val){
                    if(self.Active){
                        self._extract_trace_data(val);
                    }
                     self.Bind_IN.Par_Val=val;
                }
                ,dataflowcb_get:null
                ,is_dataflow_out:true
                ,is_dataflow_pull:false
                ,option:option
            }
        );
        //バインド切り替え時
        self.Bind_OUT.EventHandler.addListener(self.Bind_OUT.EventHandlerType.OnBindChange,function(e){
            self._bindcheck();
        });
       
        //--------------------//
        //  初期化
        //--------------------//
        //--------------------//
        //  編集用UIに公開するプロパティ
        //--------------------//
        this.PublishUIProp({prop_name:"Active",prop_show_name:"Active",prop_type:PublishUIProp_TYPE.BOOL,lowlim:null,maxlim:null,step:null});
        this.PublishUIProp({prop_name:"Log",prop_show_name:"Log",prop_type:PublishUIProp_TYPE.STRUCT_READ,lowlim:null,maxlim:null,step:null,ui_show_type:PublishUIProp_UI_SHOW_TYPE.Normal});
        // -------------------//
        //  イベントバインド
        //--------------------//
        ///破棄時のイベント（基底クラスから呼び出される）　
        this.EventHandler.addListener(self.EventHandlerType.OnDestory,function(e){});
        ////

    }
    //基底クラス継承
    _AbstractDevice_Debug_Trace.prototype = Object.create(_AbstractDevice_module_base.prototype,{constructor:{value:_AbstractDevice_Debug_Trace}});
    // -------------------//
    // section::パブリックプロパティ
    //--------------------//
    Object.defineProperty(_AbstractDevice_Debug_Trace.prototype,'Active', {
        get: function() {
            return this.PROP.Active;
        },
        set:function(val){
            if(this.PROP.Active==val){return;}
            this.PROP.Active=val;
            AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnIconViewReDrowCall,device:this});//アイコン再描画用
        }
    });
    Object.defineProperty(_AbstractDevice_Debug_Trace.prototype,'Log', {
        get: function() {
            return this._debug_str;
        },
        set:function(){
        }
    });
    // -------------------//
    // section::内部関数
    //--------------------//
    //バインダの反対側を接続先のデータタイプに書き換える
    _AbstractDevice_Debug_Trace.prototype._bindcheck=function(){
        if(this.Bind_IN.is_bind && !this.Bind_OUT.is_bind){
            this.Bind_OUT._allow_Comparator=this.Bind_IN._binding_binder_ref._allow_Comparator;
            this.Bind_OUT._type_list=this.Bind_IN._binding_binder_ref._type_list;
        }
        if(this.Bind_OUT.is_bind && !this.Bind_IN.is_bind){
            this.Bind_IN._allow_Comparator=this.Bind_OUT._binding_binder_ref._allow_Comparator;
            this.Bind_IN._type_list=this.Bind_OUT._binding_binder_ref._type_list;
        }

        //全バインダーが空の場合、デフォルトに戻す
        if(!this.Bind_IN.is_bind && !this.Bind_OUT.is_bind){
            this.Bind_IN._type_list=this.Bind_OUT._type_list=DeviceValBinder_TYPE_LIST.VAL_ALL;
            this.Bind_IN._allow_Comparator=this.Bind_OUT._allow_Comparator=DeviceValBinder_TYPE_LIST.VAL_ALL;
        }
    };
    //データのデバッグ用出力
    _AbstractDevice_Debug_Trace.prototype._extract_trace_data=function(data){
        var st=">_";
        if(data instanceof Object){
            if(data.GetValObj){
                st=JSON.stringify(data.GetValObj());
            } else {
                st=data.toString();
            }
        }else {
            st=data;
        }
        //UTL_LOG.log("trace: "+st);
        this._debug_str=st;
        this.EventHandler.fire({type:this.EventHandlerType.OnDataChange, data:st});
        //アイコン描画
        var now=+new Date();
        if(now-this._OnIconViewReDrowCallInterval_stacktime>this._OnIconViewReDrowCallInterval) {
            this._OnIconViewReDrowCallInterval_stacktime=now;
            AbstractDeviceList.EventHandler.fire({
                type: AbstractDeviceList.EventHandlerType.OnIconViewReDrowCall,
                device: this
            });
        }
    };

    /************************************************************************************
     *  section::***** Create_Strcture 構造体作成
     *  　プロパティStructArg_0〜StructArg_3の値をデフォルト値にして、
     *  　バインダー値(Bind_StructArg_0〜Bind_StructArg_3複数指定はBind_SendStructArgsString )の値を上書きして送信
         ●バインダー
             ◎OUT	(表示名:OUTPUT Strcture)	//・作成した構造体
             ◎SendTrigger	(表示名:Strcture Sending Trigger)	//・現在保持している値から構造体を作成＞送信するトリガ
             ◎SendStructArgsString	(表示名:SendStructArgsString)	//・構造体の引数にするカンマ区切りの値(このデータを受信時に構造体を生成＞送信)
             ◎Bind_StructArg_0	(表示名:SET Struct Arg 0)	//・セットする構造体の0番目の値
             ◎Bind_StructArg_1	(表示名:SET Struct Arg 1)	//・セットする構造体の1番目の値
             ◎Bind_StructArg_2	(表示名:SET Struct Arg 2)	//・セットする構造体の2番目の値
             ◎Bind_StructArg_3	(表示名:SET Struct Arg 3)	//・セットする構造体の3番目の値
         ●プロパティ
             ◎ StructureType		//・作成する構造体の種類
             ◎ StructArg_0		//・構造体の0番目の値
             ◎ StructArg_0_Trigger		//・構造体の0番目の値を受信時に構造体を生成＞送信
             ◎ StructArg_1		//・構造体の1番目の値
             ◎ StructArg_1_Trigger		//・構造体の1番目の値を受信時に構造体を生成＞送信
             ◎ StructArg_2		//・構造体の2番目の値
             ◎ StructArg_2_Trigger		//・構造体の2番目の値を受信時に構造体を生成＞送信
             ◎ StructArg_3		//・構造体の3番目の値
             ◎ StructArg_3_Trigger		//・構造体の3番目の値を受信時に構造体を生成＞送信
     ************************************************************************************/
    function _AbstractDevice_CreateStrcture(option,class_option) {
        "use strict";
        _AbstractDevice_module_base.apply(this, arguments);
        var self = this;
        //--------------------//
        //section::追加イベント
        //--------------------//
        //--------------------//
        //section::内部プロパティ
        //--------------------//
        this._stac_args=[];
        //--------------------//
        //section::保存プロパティの復帰＆デフォルト値設定
        //--------------------//
        this.PROP.StructureType=UTL_N(option.StructureType);
        this.PROP.StructArg_0=UTL_N(option.StructArg_0);this.PROP.StructArg_1=UTL_N(option.StructArg_1);this.PROP.StructArg_2=UTL_N(option.StructArg_2);this.PROP.StructArg_3=UTL_N(option.StructArg_3);
        this.PROP.StructArg_0_Trigger=Boolean(option.StructArg_0_Trigger);
        this.PROP.StructArg_1_Trigger=Boolean(option.StructArg_1_Trigger);
        this.PROP.StructArg_2_Trigger=Boolean(option.StructArg_2_Trigger);
        this.PROP.StructArg_3_Trigger=Boolean(option.StructArg_3_Trigger);

        //--------------------//
        //section::バインダーの生成
        //--------------------//

        //バインダーOUT
        self.Bind_OUT=this.CreateBinder(
            {
                binder_name:"OUT"
                ,binder_show_name:'Output Strcture'
                ,type_list: DeviceValBinder_TYPE_LIST.VAL_ALL
                ,allow_list:DeviceValBinder_TYPE_LIST.VAL_ALL
                ,dataflowcb_in:null
                ,dataflowcb_get:null
                ,is_dataflow_out:true
                ,is_dataflow_pull:false
                ,option:option
            }
        );
        self.Bind_SendTrigger=this.CreateBinder(
            {
                binder_name:"SendTrigger"
                ,binder_show_name:'Strcture Sending Trigger'
                ,type_list: DeviceValBinder_TYPE_LIST.VAL_NUMBER
                ,allow_list:DeviceValBinder_TYPE_LIST.VAL_ALL
                ,dataflowcb_in:function(val){
                    self._StructPush();
                }
                ,dataflowcb_get:null
                ,is_dataflow_out:false
                ,is_dataflow_pull:false
                ,option:option
            }
        );
        self.Bind_SendStructArgsString=this.CreateBinder(
            {
                binder_name:"SendStructArgsString"
                ,binder_show_name:'Send Struct Args String'
                ,type_list: DeviceValBinder_TYPE_LIST.VAL_STRING
                ,allow_list:DeviceValBinder_TYPE_LIST.VAL_STRING
                ,dataflowcb_in:function(val){
                    if(!val){return;}
                        var ar=String(val).split(",");
                        for(var i=0;i<ar.length;i++){
                            ar[i]=UTL_N(ar[i]);
                        }
                        self._StructPush(ar);
                 }
                ,dataflowcb_get:null
                ,is_dataflow_out:false
                ,is_dataflow_pull:false
                ,option:option
            }
        );
        self.Bind_StructArg_0=this.CreateBinder(
            {
                binder_name:"Bind_StructArg_0"
                ,binder_show_name:'Set Struct Arg 0'
                ,type_list: DeviceValBinder_TYPE_LIST.VAL_NUMBER
                ,allow_list:DeviceValBinder_TYPE_LIST.VAL_NUMBER
                ,dataflowcb_in:function(val){
                self._stac_args[0]=UTL_N(val);
                if(self.StructArg_0_Trigger){
                    self._StructPush();
                }
            }
                ,dataflowcb_get:null
                ,is_dataflow_out:false
                ,is_dataflow_pull:false
                ,option:option
            }
        );
        self.Bind_StructArg_1=this.CreateBinder(
            {
                binder_name:"Bind_StructArg_1"
                ,binder_show_name:'Set Struct Arg 1'
                ,type_list: DeviceValBinder_TYPE_LIST.VAL_NUMBER
                ,allow_list:DeviceValBinder_TYPE_LIST.VAL_NUMBER
                ,dataflowcb_in:function(val){
                self._stac_args[1]=UTL_N(val);
                if(self.StructArg_1_Trigger){
                    self._StructPush();
                }
            }
                ,dataflowcb_get:null
                ,is_dataflow_out:false
                ,is_dataflow_pull:false
                ,option:option
            }
        );
        self.Bind_StructArg_2=this.CreateBinder(
            {
                binder_name:"Bind_StructArg_2"
                ,binder_show_name:'Set Struct Arg 2'
                ,type_list: DeviceValBinder_TYPE_LIST.VAL_NUMBER
                ,allow_list:DeviceValBinder_TYPE_LIST.VAL_NUMBER
                ,dataflowcb_in:function(val){
                self._stac_args[2]=UTL_N(val);
                if(self.StructArg_2_Trigger){
                    self._StructPush();
                }
            }
                ,dataflowcb_get:null
                ,is_dataflow_out:false
                ,is_dataflow_pull:false
                ,option:option
            }
        );
        self.Bind_StructArg_3=this.CreateBinder(
            {
                binder_name:"Bind_StructArg_3"
                ,binder_show_name:'Set Struct Arg 3'
                ,type_list: DeviceValBinder_TYPE_LIST.VAL_NUMBER
                ,allow_list:DeviceValBinder_TYPE_LIST.VAL_NUMBER
                ,dataflowcb_in:function(val){
                self._stac_args[3]=UTL_N(val);
                if(self.StructArg_3_Trigger){
                    self._StructPush();
                }
            }
                ,dataflowcb_get:null
                ,is_dataflow_out:false
                ,is_dataflow_pull:false
                ,option:option
            }
        );

        //--------------------//
        //  初期化
        //--------------------//
        this._ChangeType_Bind_OUT();
        //--------------------//
        //  編集用UIに公開するプロパティ
        //--------------------//
        this.PublishUIProp({prop_name:"StructureType",prop_show_name:"Structure Type",prop_type:PublishUIProp_TYPE.ENUM,lowlim:null,maxlim:null,step:this.CREATE_STRUCTURE_TYPE});
        this.PublishUIProp({prop_name:"StructArg_0",prop_show_name:"StructArg 0",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:null,maxlim:null,step:null});
        this.PublishUIProp({prop_name:"StructArg_0_Trigger",prop_show_name:"StructArg 0 Trigger",prop_type:PublishUIProp_TYPE.BOOL,lowlim:null,maxlim:null,step:null});
        this.PublishUIProp({prop_name:"StructArg_1",prop_show_name:"StructArg 1",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:null,maxlim:null,step:null});
        this.PublishUIProp({prop_name:"StructArg_1_Trigger",prop_show_name:"StructArg 1 Trigger",prop_type:PublishUIProp_TYPE.BOOL,lowlim:null,maxlim:null,step:null});
        this.PublishUIProp({prop_name:"StructArg_2",prop_show_name:"StructArg 2",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:null,maxlim:null,step:null});
        this.PublishUIProp({prop_name:"StructArg_2_Trigger",prop_show_name:"StructArg 2 Trigger",prop_type:PublishUIProp_TYPE.BOOL,lowlim:null,maxlim:null,step:null});
        this.PublishUIProp({prop_name:"StructArg_3",prop_show_name:"StructArg 3",prop_type:PublishUIProp_TYPE.FLOAT,lowlim:null,maxlim:null,step:null});
        this.PublishUIProp({prop_name:"StructArg_3_Trigger",prop_show_name:"StructArg 3 Trigger",prop_type:PublishUIProp_TYPE.BOOL,lowlim:null,maxlim:null,step:null});

        // -------------------//
        //  イベントバインド
        //--------------------//
        ///破棄時のイベント（基底クラスから呼び出される）　
        this.EventHandler.addListener(self.EventHandlerType.OnDestory,function(e){});
        ////
    }
    //基底クラス継承
    _AbstractDevice_CreateStrcture.prototype = Object.create(_AbstractDevice_module_base.prototype,{constructor:{value:_AbstractDevice_CreateStrcture}});
    // -------------------//
    // section::定数定義
    //--------------------//
    //作成する構造体の種類
    _AbstractDevice_CreateStrcture.prototype.CREATE_STRUCTURE_TYPE={
        "Vector2":0,"Vector3":1,
        "MTLedState":2,"MTRotState":3
    };
    // -------------------//
    // section::パブリックプロパティ
    //--------------------//
    Object.defineProperty(_AbstractDevice_CreateStrcture.prototype,'StructureType', {
        get: function() {
            return this.PROP.StructureType;
        },
        set:function(val){
            this.PROP.StructureType=UTL_N(val);
            this._ChangeType_Bind_OUT();
        }
    });
    Object.defineProperty(_AbstractDevice_CreateStrcture.prototype,'StructArg_0', {
        get: function() {
            return this.PROP.StructArg_0;
        },
        set:function(val){
            this.PROP.StructArg_0=UTL_N(val);
        }
    });
    Object.defineProperty(_AbstractDevice_CreateStrcture.prototype,'StructArg_1', {
        get: function() {
            return this.PROP.StructArg_1;
        },
        set:function(val){
            this.PROP.StructArg_1=UTL_N(val);
        }
    });
    Object.defineProperty(_AbstractDevice_CreateStrcture.prototype,'StructArg_2', {
        get: function() {
            return this.PROP.StructArg_2;
        },
        set:function(val){
            this.PROP.StructArg_2=UTL_N(val);
        }
    });
    Object.defineProperty(_AbstractDevice_CreateStrcture.prototype,'StructArg_3', {
        get: function() {
            return this.PROP.StructArg_3;
        },
        set:function(val){
            this.PROP.StructArg_3=UTL_N(val);
        }
    });
    Object.defineProperty(_AbstractDevice_CreateStrcture.prototype,'StructArg_0_Trigger', {
        get: function() {
            return this.PROP.StructArg_0_Trigger;
        },
        set:function(val){
            this.PROP.StructArg_0_Trigger=val;
        }
    });
    Object.defineProperty(_AbstractDevice_CreateStrcture.prototype,'StructArg_1_Trigger', {
        get: function() {
            return this.PROP.StructArg_1_Trigger;
        },
        set:function(val){
            this.PROP.StructArg_1_Trigger=val;
        }
    });
    Object.defineProperty(_AbstractDevice_CreateStrcture.prototype,'StructArg_2_Trigger', {
        get: function() {
            return this.PROP.StructArg_2_Trigger;
        },
        set:function(val){
            this.PROP.StructArg_2_Trigger=val;
        }
    });
    Object.defineProperty(_AbstractDevice_CreateStrcture.prototype,'StructArg_3_Trigger', {
        get: function() {
            return this.PROP.StructArg_3_Trigger;
        },
        set:function(val){
            this.PROP.StructArg_3_Trigger=val;
        }
    });
    // -------------------//
    // section::内部関数
    //--------------------//
    //構造体を生成して送信 _StructPush(arg_array:省略時はバインダ受信値StructArg_nの値から生成 )
    _AbstractDevice_CreateStrcture.prototype._StructPush=function(arg_array){
        if(arg_array){
            this._stac_args=arg_array;
        }

        var ar=[ this._stac_args[0]===undefined?this.StructArg_0:this._stac_args[0]
            ,this._stac_args[1]===undefined?this.StructArg_1:this._stac_args[1]
            ,this._stac_args[2]===undefined?this.StructArg_2:this._stac_args[2]
            ,this._stac_args[3]===undefined?this.StructArg_3:this._stac_args[3]
        ];

        var rt_st;
        switch (this.StructureType){
            case this.CREATE_STRUCTURE_TYPE.Vector2:
                rt_st=new ST_Vector2(ar[0],ar[1]);
                break;
            case this.CREATE_STRUCTURE_TYPE.Vector3:
                rt_st=new ST_Vector3(ar[0],ar[1],ar[2]);
                break;
            case this.CREATE_STRUCTURE_TYPE.MTLedState:
                rt_st=new ST_MTLedState(ar[0],ar[1],ar[2],ar[3]);
                break;
            case this.CREATE_STRUCTURE_TYPE.MTRotState:
                rt_st=new ST_MTRotState(ar[0],ar[1],ar[2]);
                break;
        }
        this.Bind_OUT.Par_Val=rt_st;
    };
    //構造体変更時のBind_OUTのバインダー型の検証と変更
    _AbstractDevice_CreateStrcture.prototype._ChangeType_Bind_OUT=function(){
        switch (this.StructureType){
            case this.CREATE_STRUCTURE_TYPE.Vector2:
                this.Bind_OUT._type_list=DeviceValBinder_TYPE_LIST.DEVICE_MODULE|DeviceValBinder_TYPE_LIST.VAL_VECTOR2;
                break;
            case this.CREATE_STRUCTURE_TYPE.Vector3:
                this.Bind_OUT._type_list=DeviceValBinder_TYPE_LIST.DEVICE_MODULE|DeviceValBinder_TYPE_LIST.VAL_VECTOR3;
                break;
            case this.CREATE_STRUCTURE_TYPE.MTLedState:
                this.Bind_OUT._type_list=DeviceValBinder_TYPE_LIST.DEVICE_MODULE|DeviceValBinder_TYPE_LIST.VAL_MT_LEDSTATE;
                break;
            case this.CREATE_STRUCTURE_TYPE.MTRotState:
                this.Bind_OUT._type_list=DeviceValBinder_TYPE_LIST.DEVICE_MODULE|DeviceValBinder_TYPE_LIST.VAL_MT_ROTSTATE;
                break;
        }
        //適合しない場合は強制的に解除
       if(!this.Bind_OUT.is_binding_compatible){
           this.Bind_OUT.UnBinding();
           AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnBindLineViewReDrowCall,device:this});//バインド線再描画用
       }
    };

    /************************************************************************************
     * section::***** RecCTL RECコントロール
         ●バインダー
          ◎RecCtl	(表示名:RecCtl Input)	//・位置記憶の操作プロトコル(モジュールのRecCtlに接続)
         ●プロパティ
          ◎ ButtonLength		//・記録する個数(記録ボタンの数)
     ************************************************************************************/
    function _AbstractDevice_RecCTL (option,class_option){
        _AbstractDevice_module_base.apply(this, arguments);
            var self = this;
            //--------------------//
            //section::追加イベントハンドラ定義
            //--------------------//
            this.EventHandlerType["OnBtnLengthChange"]="OnBtnLengthChange";//UI再描画用
            this.EventHandlerType["OnPossionMemorysChange"]="OnPossionMemorysChange";//UI再描画用
            //--------------------//
            //section::内部プロパティ
            //--------------------//
            this._PossionMemorys=[];
            //--------------------//
            //section::保存プロパティの復帰＆デフォルト値設定
            //--------------------//
            this.PROP.ButtonLength=option.ButtonLength?option.ButtonLength:3;
            //--------------------//
            //section::バインダーの生成
            //--------------------//

            self.Binder_RecCtl=this.CreateBinder(
                {
                    binder_name:"RecCtl"
                    ,binder_show_name:"Position Recorder"
                    ,type_list:DeviceValBinder_TYPE_LIST.DEVICE_INPUT|DeviceValBinder_TYPE_LIST.VAL_CTL_REC
                    ,allow_list:DeviceValBinder_TYPE_LIST.VAL_CTL_REC
                    ,dataflowcb_in:function(protocol){
                        self._RsvRecCtl(protocol)
                    }
                    ,dataflowcb_get:null
                    ,is_dataflow_out:true
                    ,is_dataflow_pull:false
                    ,option:option
                }
            );
            //バインド時初期値取得
            this.Binder_RecCtl.EventHandler.addListener(self.Binder_RecCtl.EventHandlerType.OnBindChange,function(e){
                var ctl_protocol= self.Binder_RecCtl.Par_Val;
                self._RsvRecCtl(ctl_protocol);
            });
            //--------------------//
            //section::初期化
            //--------------------//
            //--------------------//
            //section::編集用UIに公開するプロパティ
            //--------------------//
            this.PublishUIProp({prop_name:"ButtonLength",prop_show_name:"Button Length",prop_type:PublishUIProp_TYPE.INT,lowlim:1,maxlim:10,step:1});
            // -------------------//
            //section::イベントバインド
            //--------------------//
            ///破棄時のイベント（基底クラスから呼び出される）　
            this.EventHandler.addListener(self.EventHandlerType.OnDestory,function(e){ });
    }
    //基底クラス継承
    _AbstractDevice_RecCTL.prototype = Object.create(_AbstractDevice_module_base.prototype,{constructor:{value:_AbstractDevice_RecCTL}});
    // -------------------//
    // section::定数定義
    //--------------------//
    // -------------------//
    // section::パブリックプロパティ
    //--------------------//
    Object.defineProperty(_AbstractDevice_RecCTL.prototype,'ButtonLength', {
        get: function() {
            return this.PROP.ButtonLength;
        },
        set:function(val){
            this.PROP.ButtonLength=val;
            this.EventHandler.fire({type:this.EventHandlerType.OnBtnLengthChange,ButtonLength:val});//UI再描画用
        }
    });

    // -------------------//
    // section::内部関数
    //--------------------//
    //記録データ受信
    _AbstractDevice_RecCTL.prototype._RsvRecCtl=function(protocol){
        if(protocol instanceof ST_NotifyProtocol && protocol.NotifyName==="PossionMemorys"){
            this._PossionMemorys = protocol.Val instanceof Array?protocol.Val:[];
            this.EventHandler.fire({
                type: this.EventHandlerType.OnPossionMemorysChange,
                PossionMemorys: this._PossionMemorys
            });//UI再描画用
        }
    };

    //RECボタンの押下
    _AbstractDevice_RecCTL.prototype._SendRecBtnPush=function(num){
        this.Binder_RecCtl.Par_Val=new ST_CtlProtocol("SetRecPosition",UTL_N(num));
    };
    // //RECボタン長押しで記録消去
    // _AbstractDevice_RecCTL.prototype._SendRecBtnHold=function(num){
    //     this.Binder_RecCtl.Par_Val=new ST_CtlProtocol("SetClearPosition",UTL_N(num));
    // };
    //再生ボタンの押下
    _AbstractDevice_RecCTL.prototype._SendPlayBtnPush=function(num){
        this.Binder_RecCtl.Par_Val=new ST_CtlProtocol("SetPlayPosition",UTL_N(num));
    };

    //位置初期化ボタン
    _AbstractDevice_RecCTL.prototype._SendClearPossionBtnPush=function(){
        this.Binder_RecCtl.Par_Val=new ST_CtlProtocol("SetResetPosition",true);
    };
    //全消去ボタン
    _AbstractDevice_RecCTL.prototype._SendClearMemoryBtnPush=function(){
        this.Binder_RecCtl.Par_Val=new ST_CtlProtocol("SetClearPosition",null);
    };


    /************************************************************************************
     * section::***** inputctl_ROSWebSoc ROSソケット
     ●バインダー
     ◎ SubscribeVal	(表示名:SubscribeVal)	//受信する値
     ◎ PublishVal	(表示名:PublishVal)	//送信する値
     ●プロパティ
     ◎ SocketAddress //ソケットのアドレス[ws://]から記述
     ◎ ValueTypeIdx   //受信するデータ型
     ◎ SubscribeTopic       //受信するトピック名
     ◎ PublishTopic         //送信するトピック名

     ************************************************************************************/
    function _AbstractDevice_inputctl_ROSWebSoc(option,class_option) {
        "use strict";
        _AbstractDevice_inputctl_base.apply(this, arguments);
        var self = this;

        //--------------------//
        //section::追加イベントハンドラ定義
        //--------------------//
        this.EventHandlerType["SocketSocketStatusMessageChange"]="SocketSocketStatusMessageChange";
        this.EventHandlerType["SubscribeValChange"]="SubscribeValChange";
        this.EventHandlerType["PublishValChange"]="PublishValChange";
        //--------------------//
        //section::内部プロパティ
        //--------------------//
        this._ros=new ROSLIB.Ros();
        this._SubscribeTopic_ROSObj=null;
        this._PublishTopic_ROSObj=null;
        this._SocketStatusMessage="";
        //--------------------//
        //section::保存プロパティの復帰＆デフォルト値設定
        //--------------------//
        this.PROP.SocketAddress=option.SocketAddress?option.SocketAddress:'127.0.0.1:9090';
        this.PROP.SubscribeTopic=option.SubscribeTopic?option.SubscribeTopic:'';
        this.PROP.ValueTypeIdx=(option.ValueTypeIdx===undefined?this.VALUE_TYPE_IDX.NUMBER:UTL_N(option.ValueTypeIdx));
        this.PROP.PublishTopic=option.PublishTopic?option.PublishTopic:'';

        //ステータスは接続状態に使用
        this.status=this._ros.isConnected;
        //--------------------//
        //section::バインダーの生成
        //--------------------//
        self.Binder_Val=this.CreateBinder(
            {
                binder_name:"Value"
                ,binder_show_name:"Value"
                ,type_list:DeviceValBinder_TYPE_LIST.VAL_ALL
                ,allow_list:DeviceValBinder_TYPE_LIST.VAL_ALL
                ,dataflowcb_in:function(val){
                if(self._ros.isConnected&&self._PublishTopic_ROSObj){
                    var ros_json = self._ConvartStrct_ToROS(val);
                    self._PublishTopic_ROSObj.publish(ros_json);
                }
            }
                ,dataflowcb_get:null
                ,is_dataflow_out:true
                ,is_dataflow_pull:false
                ,option:option
            }
        );

        //--------------------//
        //section::編集用UIに公開するプロパティ
        //--------------------//
        this.PublishUIProp({prop_name:"SocketStatusMessage",prop_show_name:"Socket Status Message",prop_type:PublishUIProp_TYPE.STRUCT_READ,lowlim:null,maxlim:null,step:null});
        this.PublishUIProp({prop_name:"SocketConnect",prop_show_name:"ROS Socket Connect",prop_type:PublishUIProp_TYPE.SEND_NULL,lowlim:null,maxlim:null,step:null});
        this.PublishUIProp({prop_name:"SocketClose",prop_show_name:"ROS Socket Close",prop_type:PublishUIProp_TYPE.SEND_NULL,lowlim:null,maxlim:null,step:null});
        this.PublishUIProp({prop_name:"SocketAddress",prop_show_name:"ROS Socket Address",prop_type:PublishUIProp_TYPE.STRING,lowlim:null,maxlim:null,step:null});
        this.PublishUIProp({prop_name:"ValueTypeIdx",prop_show_name:"Value Type",prop_type:PublishUIProp_TYPE.ENUM,lowlim:null,maxlim:null,step:this.VALUE_TYPE_IDX});
        this.PublishUIProp({prop_name:"SubscribeTopic",prop_show_name:"Subscribe Topic Name",prop_type:PublishUIProp_TYPE.STRING,lowlim:null,maxlim:null,step:null});
        this.PublishUIProp({prop_name:"PublishTopic",prop_show_name:"Publish Topic Name",prop_type:PublishUIProp_TYPE.STRING,lowlim:null,maxlim:null,step:null});


        //--------------------//
        //section::初期化
        //--------------------//
        this._ChangeType_Binder_val();
        // -------------------//
        //section::イベントバインド ROS
        //--------------------//
        this._ros.on('error', function(error) {
            self.status=false;
           self.SocketStatusMessage="WebSoc error!";
            UTL_LOG("ROSWebSoc error");
        });
        this._ros.on('connection', function(error) {
            self.status=true;
            self.SocketStatusMessage="WebSoc connection!";
            UTL_LOG("ROSWebSoc connection");
        });
        this._ros.on('close', function(error) {
            self.status=false;
            self.SocketStatusMessage="WebSoc close!";
            UTL_LOG("ROSWebSoc close");
        });
        // -------------------//
        //section::イベントバインド
        //--------------------//
        ///破棄時のイベント（基底クラスから呼び出される）　
        this.EventHandler.addListener(self.EventHandlerType.OnDestory,function(e){
            if(self._ros.isConnected){
                self._ros.close();
            }
        });


    }
    //基底クラス継承
    _AbstractDevice_inputctl_ROSWebSoc.prototype = Object.create(_AbstractDevice_inputctl_base.prototype,{constructor:{value:_AbstractDevice_inputctl_ROSWebSoc}});
    // -------------------//
    // section::定数定義
    //--------------------//
    //接続するデータの種類
    _AbstractDevice_inputctl_ROSWebSoc.prototype.VALUE_TYPE_IDX={
        "BOOL":0
        ,"NUMBER":1
        ,"STRING":2
        ,"Vector2":3
        ,"Vector3":4
      //  ,"MTLedState":5
        ,"MTRotState":6
        ,"CtlProtocol":7
       // ,"NotifyProtocol":8
    };

    /*************************
     *  型変換リスト
     *  KM型          ROS型
     *  BOOL         'std_msgs/Bool'
     *　NUMBER    'std_msgs/Float64'
     *  STRING    'std_msgs/String'
     *  Vector2  'std_msgs/Float64MultiArray'
     *  Vector3  'std_msgs/Float64MultiArray'
     *  MTLedState  'std_msgs/Float64MultiArray'
     *  MTRotState  'std_msgs/Float64MultiArray'
     *  CtlProtocol  'std_msgs/Float64MultiArray'
     *************************/
    _AbstractDevice_inputctl_ROSWebSoc.prototype.VALUE_TYPE=[
        [DeviceValBinder_TYPE_LIST.VAL_BOOL,'std_msgs/Bool']//BOOL
        ,[DeviceValBinder_TYPE_LIST.VAL_NUMBER,'std_msgs/Float64']//NUMBER
        ,[DeviceValBinder_TYPE_LIST.VAL_STRING,'std_msgs/String']//STRING
        ,[DeviceValBinder_TYPE_LIST.VAL_VECTOR2,'std_msgs/Float64MultiArray']//Vector2
        ,[DeviceValBinder_TYPE_LIST.VAL_VECTOR3,'std_msgs/Float64MultiArray']//Vector3
       ,[DeviceValBinder_TYPE_LIST.VAL_MT_LEDSTATE,'std_msgs/Float64MultiArray']//MTLedState
        ,[DeviceValBinder_TYPE_LIST.VAL_MT_ROTSTATE,'std_msgs/Float64MultiArray']//MTRotState
        ,[DeviceValBinder_TYPE_LIST.VAL_CTL_MT,'std_msgs/String']//CtlProtocol
        //,[DeviceValBinder_TYPE_LIST.VAL_OBJ,'std_msgs/String']//NotifyProtocol
       // ,[DeviceValBinder_TYPE_LIST.VAL_OBJ,'std_msgs/String']//VAL_OBJ
    ];
    // -------------------//
    // section::パブリックプロパティ
    //--------------------//

    Object.defineProperty(_AbstractDevice_inputctl_ROSWebSoc.prototype,'SocketStatusMessage', {
        get: function() {return this._SocketStatusMessage;},
        set:function(val){
            this._SocketStatusMessage=val;
            this.EventHandler.fire({type:this.EventHandlerType.SocketSocketStatusMessageChange,SocketSocketStatusMessage:val});//UI再描画用
        }
    });
    Object.defineProperty(_AbstractDevice_inputctl_ROSWebSoc.prototype,'SocketAddress', {
        get: function() {return this.PROP.SocketAddress;},
        set:function(val){this.PROP.SocketAddress=val;}
    });
    Object.defineProperty(_AbstractDevice_inputctl_ROSWebSoc.prototype,'ValueTypeIdx', {
        get: function() {return this.PROP.ValueTypeIdx;},
        set:function(val){
            this.PROP.ValueTypeIdx=val;
            this._ChangeType_Binder_val();
        }
    });
    Object.defineProperty(_AbstractDevice_inputctl_ROSWebSoc.prototype,'SubscribeTopic', {
        get: function() {return this.PROP.SubscribeTopic;},
        set:function(val){this.PROP.SubscribeTopic=val;}
    });
    Object.defineProperty(_AbstractDevice_inputctl_ROSWebSoc.prototype,'PublishTopic', {
        get: function() {return this.PROP.PublishTopic;},
        set:function(val){this.PROP.PublishTopic=val;}
    });
    //ソケットへの接続
    Object.defineProperty(_AbstractDevice_inputctl_ROSWebSoc.prototype,'SocketConnect', {
        get: function() { return null;},
        set:function(val){
            var self=this;
            if(!this._ros.isConnected){
                this._ros.connect("ws://"+this.SocketAddress);
                if(this.SubscribeTopic){
                    this._SubscribeTopic_ROSObj= new ROSLIB.Topic({
                        ros : self._ros,
                        name : self.SubscribeTopic,
                        messageType : self.VALUE_TYPE[self.ValueTypeIdx][1]//ValueTypeに対応するROSデータ型
                    });

                    this._SubscribeTopic_ROSObj.subscribe(function(message) {
                        self.Binder_Val.Par_Val=self._ConvartStrct_FromROS(message);
                    });
                }
                if(this.PublishTopic){
                    this._PublishTopic_ROSObj = new ROSLIB.Topic({
                        ros : self._ros,
                        name : self.PublishTopic,
                        messageType : self.VALUE_TYPE[self.ValueTypeIdx][1]//ValueTypeに対応するROSデータ型
                    });
                }
            }
        }
    });
    //ソケットの切断
    Object.defineProperty(_AbstractDevice_inputctl_ROSWebSoc.prototype,'SocketClose', {
        get: function() { return null;},
        set:function(val){
            if(this._ros.isConnected){
                this._ros.close();
            }
        }
    });


    // -------------------//
    // section::内部関数
    //--------------------//

    //KMからROSへの送信時の型変換
    _AbstractDevice_inputctl_ROSWebSoc.prototype._ConvartStrct_ToROS=function(val){
        var conv;
        //info::CtlProtocolと併送される notifyプロトコルのみJSONStringに変換
        if(this.ValueTypeIdx==this.VALUE_TYPE_IDX.CtlProtocol){
            conv=JSON.stringify(this._ConvartStrctFormat_ToROS(val));
        }else{
            conv=this._ConvartStrctFormat_ToROS(val)[1];//値のみ送信
        }
        return new ROSLIB.Message({data :conv});
    };
    //ROSからKMへの受信時の型変換
    _AbstractDevice_inputctl_ROSWebSoc.prototype._ConvartStrct_FromROS=function(Message){
        //console.log(Message);
        var data=Message.data;
        var conv=null;

        switch (this.ValueTypeIdx){
            case this.VALUE_TYPE_IDX.BOOL:
                conv=this._ConvartStrctFormat_FromROS(['BOOL',data]);
                break;
            case this.VALUE_TYPE_IDX.NUMBER:
                conv=this._ConvartStrctFormat_FromROS(['NUMBER',data]);
                break;
            case this.VALUE_TYPE_IDX.STRING:
                conv=this._ConvartStrctFormat_FromROS(['STRING',data]);
                break;
            case this.VALUE_TYPE_IDX.Vector2:
                conv=this._ConvartStrctFormat_FromROS(['VECTOR2',data]);
                break;
            case this.VALUE_TYPE_IDX.Vector3:
                conv=this._ConvartStrctFormat_FromROS(['VECTOR3',data]);
                break;
            case this.VALUE_TYPE_IDX.MTLedState:
                conv=this._ConvartStrctFormat_FromROS(['LEDSTATE',data]);
                break;
            case this.VALUE_TYPE_IDX.MTRotState:
                conv=this._ConvartStrctFormat_FromROS(['ROTSTATE',data]);
                break;
            case this.VALUE_TYPE_IDX.CtlProtocol:
                var ss=UTL_TRY_JSON_parse(data);
                //info::CtlProtocolと併送されるnotifyプロトコルのみ例外的に抽出
                if(ss instanceof Object && ss.NotifyName){
                    var notifyVal=this._ConvartStrctFormat_FromROS(ss.Val);
                    conv=new ST_NotifyProtocol(ss.NotifyName,notifyVal,ss.originID);
                }else{
                    ss=ss instanceof Array?ss:[];
                    var cmnd=ss[0];
                    var data=ss[1];
                    var val=this._ConvartStrctFormat_FromROS(data);
                    conv=new ST_CtlProtocol(cmnd,val,null);
                }
                break;
        }
        return conv;

    };
    //プリミティブ・構造体の復元(ROS型-->KM型)
    //in_data型の構造体のフォーマット in_data['VECTOR3',[position,speed,torque]]
    _AbstractDevice_inputctl_ROSWebSoc.prototype._ConvartStrctFormat_FromROS=function(in_data){
        if(!(in_data instanceof Array)){return null;}
        var type=in_data[0];
        var ini_v=in_data[1];
        var r_val=null;
        switch (type){
            case 'BOOL':
                r_val=Boolean(ini_v);
                break;
            case 'NUMBER':
                r_val=UTL_N(ini_v);
                break;
            case 'STRING':
                r_val=ini_v;
                break;
            case 'VECTOR2':
                var d=ini_v instanceof Array?ini_v:[];
                r_val=new ST_Vector2(d[0],d[1]);
                break;
            case 'VECTOR3':
                var d=ini_v instanceof Array?ini_v:[];
                r_val=new ST_Vector3(d[0],d[1],d[2]);
                break;
            case 'LEDSTATE':
                var d=ini_v instanceof Array?ini_v:[];
                r_val=new ST_MTLedState(d[0],d[1],d[2],d[3]);
                break;
            case 'ROTSTATE':
                var d=ini_v instanceof Array?ini_v:[];
                r_val=new ST_MTRotState(d[0],d[1],d[2]);
                break;
            default:
                r_val=(ini_v instanceof Array||ini_v instanceof Object)?ini_v:null;
                break;
        }
        return r_val;
    };
    //プリミティブ・構造体のシリアライズ(KM型-->ROS型)
    _AbstractDevice_inputctl_ROSWebSoc.prototype._ConvartStrctFormat_ToROS=function(in_data){
        var r=[];
        if(in_data==null){return;}
            switch (in_data.constructor){
                case Boolean:
                    r=['BOOL',Boolean(in_data)];
                    break;
                case Number:
                    r=['NUMBER',UTL_N(in_data)];
                    break;
                case String:
                    r=['STRING',in_data];
                    break;
                case ST_Vector2:
                    r=['VECTOR2',[in_data.x,in_data.y]];
                    break;
                case ST_Vector3:
                    r=['VECTOR3',[in_data.x,in_data.y,in_data.z]];
                    break;
                case ST_MTLedState:
                    r=['LEDSTATE',[in_data.State,in_data.ColorR,in_data.ColorG,in_data.ColorB]];
                    break;
                case ST_MTRotState:
                    r=['ROTSTATE',[in_data.Position,in_data.Speed,in_data.Torque]];
                    break;
                case ST_CtlProtocol:
                    r=[in_data.Command,this._ConvartStrctFormat_ToROS(in_data.Val)];
                    break;
                case ST_NotifyProtocol:
                    r={NotifyName:in_data.NotifyName,Val:this._ConvartStrctFormat_ToROS(in_data.Val),originID:in_data.originID};
                    break;
                case Array:
                    r=['ARRAY',in_data];
                    break;
                case Object:
                    r=['OBJECT',in_data];
                    break;
            }
        return r;
    };

    //データ型 変更時のBind_OUTのバインダー型の検証と変更
    _AbstractDevice_inputctl_ROSWebSoc.prototype._ChangeType_Binder_val=function(){
        this.Binder_Val._type_list=this.VALUE_TYPE[this.ValueTypeIdx][0];

        //適合しない場合は強制的に解除
        if(!this.Binder_Val.is_binding_compatible){
            this.Binder_Val.UnBinding();
            AbstractDeviceList.EventHandler.fire({type:AbstractDeviceList.EventHandlerType.OnBindLineViewReDrowCall,device:this});//バインド線再描画用
        }
    };

    /************************************************************************************
     * END 仮想デバイス定義
     ************************************************************************************/


    /************************************************************************************
     * 仮想デバイス生成メソッド・定数
     ************************************************************************************/
    // -------------------//
    // プライベート領域
    //--------------------//
    //デバイスのクラスリスト //todo::ここでハードの状態を検出して、ハード毎の互換のあるクラスを自動生成（ロード）する
    var _ClassOptions=[
        // -------------------//
        //  BLEモジュール
        // -------------------//
        //info::[device_name:"ble_KeiganMotor"]は仮想デバイス追加、KM同士をD&Dの処理等、モーターレジスタ設定表示欄html 各種jsでハードコードされている
        // info::ble_device_name はモーターに設定されている名称を取得“Keigan Motor XXX#RGB”　XXX:固有値 #RGB:LED色
        {device_name:"ble_KeiganMotor",ble_device_name:"null",ble_service_uuid:"f140ea35-8936-4d35-a0ed-dfcd795baa8c",abs_device:_AbstractDevice_Keigan_Motor,DeviceType:AbstractDeviceList_DeviceType.BLE,
            //can_motor_grid_swap:グリッド上でドラッグして相互にモーター入れ替え(同じシリーズ)が可能か。(KeiganMotorシリーズのみ)
            UIinfo:{ico:'img/deviceico/ble_KeiganMotor.svg',name:'Keigan Motor',info:WORD('ble_KeiganMotor_info'),can_motor_grid_swap:true}}
        
        // -------------------//
        //  その他モジュール
        // -------------------//
        ,{device_name:"SingleWheelCar_Assembly",abs_device:_AbstractDevice_SingleWheelCar_Assembly,DeviceType:AbstractDeviceList_DeviceType.Robot,
            UIinfo:{ico:'img/deviceico/SingleWheelCar_Assembly.svg',name:'Single Axis Robot',info:WORD('SingleWheelCar_Assembly_info')}}

        ,{device_name:"TwoWheelCar_Assembly",abs_device:_AbstractDevice_TwoWheelCar_Assembly,DeviceType:AbstractDeviceList_DeviceType.Robot,
            UIinfo:{ico:'img/deviceico/TwoWheelCar_Assembly.svg',name:'2-Wheeled Car',info:WORD('TwoWheelCar_Assembly_info')}}

        ,{device_name:"PanTiltGimbal_Assembly",abs_device:_AbstractDevice_PanTiltGimbal_Assembly,DeviceType:AbstractDeviceList_DeviceType.Robot,
            UIinfo:{ico:'img/deviceico/PanTiltGimbal_Assembly_v2.svg',name:'2-Axis Gimbal',info:WORD('PanTiltGimbal_Assembly_info')}}

        ///ターンテーブル
        ,{device_name:"SensingTurnTable_Assembly",abs_device:_AbstractDevice_SensingTurnTable_Assembly,DeviceType:AbstractDeviceList_DeviceType.Robot,
            UIinfo:{ico:'img/deviceico/SensingTurnTable_Assembly.svg',name:'Experimental: Turntable',info:WORD('SensingTurnTable_Assembly_info')}}

        ,{device_name:"val_branch",abs_device:_AbstractDevice_val_branch,DeviceType:AbstractDeviceList_DeviceType.Module,
            UIinfo:{ico:'img/deviceico/value_branch.svg',name:'Value Branch',info:WORD('val_branch_info')}}

        ///モーター変速機
        ,{device_name:"MotorTransmission_Assembly",abs_device:_AbstractDevice_MotorTransmission_Assembly,DeviceType:AbstractDeviceList_DeviceType.Module,
            UIinfo:{ico:'img/deviceico/MotorTransmission_Assembly.svg',name:'Gear',info:WORD('MotorTransmission_Assembly_info')}}

        ///デバッグトレース
        ,{device_name:"Debug_Trace",abs_device:_AbstractDevice_Debug_Trace,DeviceType:AbstractDeviceList_DeviceType.Module,
            UIinfo:{ico:'img/deviceico/Debug_Trace.svg',name:'Debug Trace',info:WORD('Debug_Trace_info')}}
        ///構造体作成
        ,{device_name:"Create_Strcture",abs_device:_AbstractDevice_CreateStrcture,DeviceType:AbstractDeviceList_DeviceType.Module,
            UIinfo:{ico:'img/deviceico/Create_Strcture.svg',name:'Create Strcture',info:WORD('Create_Strcture_info')}}

        //todo::ジャイロ実装途中
        // ,{device_name:"inputctl_gyroscope",abs_device:_AbstractDevice_inputctl_gyroscope,DeviceType:AbstractDeviceList_DeviceType.Input,
        //     UIinfo:{ico:'img/deviceico/slc_ctl_gyro.svg',name:'GyroScope',info:WORD('inputctl_gyroscope_info')}}

        ,{device_name:"RecCTL",abs_device:_AbstractDevice_RecCTL,DeviceType:AbstractDeviceList_DeviceType.Input,
            UIinfo:{ico:'img/deviceico/RecCTL.svg',name:'Position Recorder',info:WORD('RecCTL_info')}}

        ,{device_name:"inputctl_XYPad",abs_device:_AbstractDevice_inputctl_XYPad,DeviceType:AbstractDeviceList_DeviceType.Input,
            UIinfo:{ico:'img/deviceico/inputctl_XYPad.svg',name:'XY Pad',info:WORD('inputctl_XYPad_info')}}

        ,{device_name:"inputctl_PushBtn",abs_device:_AbstractDevice_inputctl_PushBtn,DeviceType:AbstractDeviceList_DeviceType.Input,
            UIinfo:{ico:'img/deviceico/inputctl_PushBtn.svg',name:'Push Button',info:WORD('inputctl_PushBtn_info')}}

        ,{device_name:"inputctl_NumberSlider",abs_device:_AbstractDevice_inputctl_NumberSlider,DeviceType:AbstractDeviceList_DeviceType.Input,
            UIinfo:{ico:'img/deviceico/inputctl_NumberSlider.svg',name:'Number Slider',info:WORD('inputctl_NumberSlider_info')}}

        ,{device_name:"inputctl_MTControlKit",abs_device:_AbstractDevice_inputctl_MTControlKit,DeviceType:AbstractDeviceList_DeviceType.Input,
            UIinfo:{ico:'img/deviceico/inputctl_MTControlKit.svg',name:'Direct Motor Control Kit',info:WORD('inputctl_MTControlKit_info')}}

        //ROSソケット
        ,{device_name:"inputctl_ROSWebSoc",abs_device:_AbstractDevice_inputctl_ROSWebSoc,DeviceType:AbstractDeviceList_DeviceType.Input,
            UIinfo:{ico:'img/deviceico/inputctl_ROSWebSoc.svg',name:'ROSbridge Websocket Client',info:WORD('inputctl_ROSWebSoc_info')}}
    ];

    // -------------------//
    // パブリック領域
    //--------------------//
    var publicobj = {
        ClassOptions:_ClassOptions,
        /**
         * section::デバイス名から抽象デバイスを検索＞生成して返す
         * @param device_name
         * @param option 初期化時に引き渡す引数obj
         *     option{
         *     "プロパティ名":初期化時に渡すオブジェクトの各種PROPの保存値
         *     is_private:内部生成用（主にバインダーを非公開にする）フラグ
         *     }
         * @constructor
         */
        CreateDeviceForSearch:function(device_name,option,successCB, failureCB){
            var absdev;
            var ClassList;
            for(var i=0;i<_ClassOptions.length;i++){
                if(_ClassOptions[i].device_name===device_name){
                    absdev=_ClassOptions[i].abs_device;
                    ClassList=_ClassOptions[i];
                    break;
                }
            }
            if(absdev){
                if(successCB){
                    var r= new absdev(option,ClassList);
                    //デバイス名が固有になるようにサブ名を設定　IsUniqueSubName
                    if(!r.device_sub_name){
                        //重複名は2から接尾辞を付加
                        if(!DeviceController.IsUniqueSubName(r.device_name,r.device_sub_name)){
                            for(var i=2;i<100;i++){//自動連番の上限 100
                                if(DeviceController.IsUniqueSubName(r.device_name,i.toString())){
                                    r.device_sub_name=i.toString();
                                    break;
                                }
                            }
                        }
                    }
                    successCB(r);
                }
            }else{
                if(failureCB){
                    failureCB(device_name);
                }
            }
        },

        /**
         * section::BLEデバイス名から抽象クラスを検索＞生成して返す
         * @param device_name
         * @param option 初期化時に引き渡す引数obj
         *     option{
         *     "プロパティ名":初期化時に渡すオブジェクトの各種PROPの保存値
         *     is_private:内部生成用（主にバインダーを非公開にする）フラグ
         *     }
         * @constructor
         */
        CreateDeviceForSearchBLE:function(BLEDeviceInfo,successCB, failureCB){
            //BLEのメインサービスUUIDから該当する仮想デバイス名を検索し、仮想デバイスを生成
            var device_name;
            for(var i=0;i<this.ClassOptions.length;i++){
                if(this.ClassOptions[i].ble_service_uuid===BLEDeviceInfo.main_service_uuid){
                    device_name=this.ClassOptions[i].device_name;
                }
            }
            if(!device_name){
                if(failureCB){
                    failureCB('ble_device_name:'+BLEDeviceInfo.device_name+' ServiceUUID:'+BLEDeviceInfo.main_service_uuid);
                }
            }else{
                this.CreateDeviceForSearch(device_name,{ble_device_name:BLEDeviceInfo.device_name,device_uuid:BLEDeviceInfo.device_uuid},successCB, failureCB);
            }
        },
        //編集画面のモジュールアイコンの各種描画要求
       // EventHandlerType:{"OnBindLineViewReDrowCall": "OnBindLineViewReDrowCall","OnIconViewReDrowCall": "OnIconViewReDrowCall","OnIconViewLedStateReDrowCall":"OnIconViewLedStateReDrowCall","OnIconViewRemoveDrowCall":"OnIconViewRemoveDrowCall"},
        EventHandlerType:{"OnBindLineViewReDrowCall": "OnBindLineViewReDrowCall","OnIconViewReDrowCall": "OnIconViewReDrowCall","OnIconViewRemoveDrowCall":"OnIconViewRemoveDrowCall"},

        EventHandler:new EventTarget()

        ///
    };
    return publicobj;
}());